define("cq/feature/viewquestion",
    [
        "jquery",
        "cq/util/events",
        "cq/util/analytics",
        "cq/util/form",
        "cq/api/question",
        "cq/api/question-service",
        "cq/feature/viewquestion/comments",
        "cq/feature/viewquestion/answer",
        "cq/feature/viewquestion/accept-answer-link",
        "cq/feature/viewquestion/topic-suggestions",
        "cq/widget/sidebarquestion",
        "exports"
    ],
    function (
            $,
            events,
            analytics,
            formUtil,
            Question,
            questionService,
            comments,
            answer,
            acceptAnswerLink,
            topicSuggestions,
            SideBarQuestion,
            exports) {


        /**
         * Scrolls the selected answer into view if we have a direct link to an answer, e.g.:
         * http://localhost:8080/confluence/questions/2588870/answers/2588937
         *
         * Highlights this answer if we are *not* in edit mode for this answer, e.g.
         * http://localhost:8080/confluence/questions/2588870/answers/2588937/edit
         *
         */
        function showSelectedAnswer() {
            var answerIdMatch = /answers\/(\d+)(\/edit)?/.exec(location.pathname);
            if (!answerIdMatch || answerIdMatch.length < 3) {
                return;
            }

            var answerId = answerIdMatch[1];
            var editAnswerMode = answerIdMatch[2];

            var $el = $("[data-content-id=" + answerId + "]");
            if (!$el[0]) {
                // could not found the answer to scroll to highlight
                return;
            }

            $el[0].scrollIntoView();

            if (!editAnswerMode) {
                var $highlightedEl = $(".cq-answer-content", $el);
                $highlightedEl.addClass("cq-highlighted");
                setTimeout(function() {
                    $highlightedEl.removeClass("cq-highlighted");
                }, 1000);
            }
        }

        /**
         * Scrolls the answer editor into view if we have a direct link to answerquestion.action, e.g.:
         * http://localhost:8080/confluence/cq/answerquestion.action
         */
        function focusAnswerEditor() {
            var answerAction = /answerquestion\.action/.exec(location.pathname);
            if (!answerAction) {
                return;
            }

            var $el = $(".cq-answer-form");
            $el[0].scrollIntoView();
        }

        function makeDeleteDialogs() {

            function makeDeleteDialog(options) {
                var dialog = AJS.InlineDialog($(".cq-delete-" + options.type), "delete-content", function (content, trigger, showPopup) {
                    content.html(CQ.Templates.feature.viewquestion.deleteContent({
                        title: options.title,
                        body: options.body
                    }));

                    content.find(".cq-delete-cancel").on("click", function () {
                        dialog.hide();
                    });

                    content.find(".cq-delete-submit").on("click", function () {
                        var dfd = $.ajax({
                            type: "DELETE",
                            url: options.getDeleteUrl($(trigger))
                        });

                        // set disabled state
                        var $this = $(this);
                        $this.attr("disabled", "disabled").attr("data-ajax-request", true);
                        dfd.always(function () {
                            $this.attr("disabled", "").attr("data-ajax-request", false);
                        });

                        dfd.done(function () {
                            analytics.trackEvent(options.type, "delete");
                            dialog.hide();
                        });

                        options.onSubmit(dfd, $(trigger));
                    });

                    showPopup();
                }, {
                    useLiveEvents: true
                });

                return dialog;
            }

            makeDeleteDialog({
                type: "question",
                title: AJS.I18n.getText("cq.question.delete.title"),
                body: AJS.I18n.getText("cq.question.delete.body"),
                getDeleteUrl: function($trigger) {
                    var questionId = $trigger.data("question-id");
                    return AJS.contextPath() + "/rest/questions/1.0/questions/" + questionId
                },
                onSubmit: function(dfd, $trigger) {
                    dfd.done(function () {
                        window.location = $trigger.attr("href");
                    });
                }
            });

            makeDeleteDialog({
                type: "answer",
                title: AJS.I18n.getText("cq.question.answer.delete.title"),
                body: AJS.I18n.getText("cq.question.answer.delete.body"),
                getDeleteUrl: function($trigger) {
                    var questionId = $trigger.data("question-id");
                    var answerId = $trigger.data("answer-id");
                    return AJS.contextPath() + "/rest/questions/1.0/questions/" + questionId + "/answers/" + answerId;
                },
                onSubmit: function(dfd, $trigger) {
                    dfd.done(function () {
                        var currentPage = $(".answers-paging li.aui-nav-selected").attr("data-index");
                        var noOfAnswers = $(".cq-answers li.cq-answer").length;
                        // if we delete the last answer in current page
                        // we need to reload page using previous page
                        if(currentPage && currentPage > 1 && noOfAnswers === 1) {
                            var previousLink = $(".answers-paging li.aui-nav-previous a").attr("href");
                            if(previousLink) {
                                window.location.href = previousLink;
                                return;
                            }
                        }

                        window.location.reload();
                    });
                }

            });

            makeDeleteDialog({
                type: "comment",
                title: AJS.I18n.getText("cq.question.comment.delete.title"),
                body: AJS.I18n.getText("cq.question.comment.delete.body"),
                getDeleteUrl: function($trigger) {
                    var questionId = $trigger.data("question-id");
                    var commentId = $trigger.data("comment-id");
                    return AJS.contextPath() + "/rest/questions/1.0/questions/" + questionId + "/comments/" + commentId;
                },
                onSubmit: function(dfd) {
                    dfd.done(function () {
                        window.location.reload();
                    });
                }
            });
        }

        function makeConvertToCommentDialogs()
        {
            function getTargets(excludeId) {
                var targets = [];

                //We always want the question to appear as the first target.
                targets.push({
                    value: AJS.Meta.get("question-id"),
                    text: AJS.I18n.getText("cq.question.answer.convert-to-answer.option.question")
                });

                $(".cq-answers .cq-answer").each(function () {
                    var $this = $(this);
                    var id = $this.data("content-id");
                    var name = $.trim($this.find(".cq-content").text());
                    if (name.length > 33) {
                        name = name.substring(0, 30) + "...";
                    }
                    var username = $this.find(".cq-userlink").first().text();

                    if (id != excludeId) {
                        targets.push({
                            value: id,
                            text: username + ": " + name
                        });
                    }
                });

                return targets;
            }

            var dialog = AJS.InlineDialog(
                    $(".cq-convert-answer-to-comment"), "convert-answer-to-comment", function (content, trigger, showPopup) {
                        content.html(CQ.Templates.feature.viewquestion.convertAnswerToComment({
                            targets: getTargets($(trigger).data("answer-id"))
                        }));

                        content.find("[name=cq-convert-answer-parent-selector]").auiSelect2();
                        content.find(".cq-convert-answer-to-comment-submit").on("click", function () {
                            var questionId = $(trigger).data("question-id");
                            var answerId = $(trigger).data("answer-id");
                            var targetId = dialog.find(".cq-convert-answer-form select").val();

                            //Disable confirm button
                            $(this).attr("disabled", "disabled");

                            var dfd = $.ajax({
                                type: "POST",
                                url: AJS.contextPath() + "/rest/questions/1.0/questions/" + questionId + "/answers/" + answerId + "/convert" + "?targetId=" + targetId
                            });

                            dfd.done(function (comment) {
                                analytics.trackEvent("answer", "convertedtocomment");
                                window.location.href = AJS.contextPath() + comment.url;
                            });
                        });

                        content.find(".cq-convert-answer-to-comment-cancel").on("click", function () {
                            dialog.hide();
                        });

                        showPopup();
                    }
            );
        }

        function makeConvertToAnswerDialogs() {
            var dialog = AJS.InlineDialog(
                    $(".cq-convert-comment-to-answer"), "convert-comment-to-answer", function (content, trigger, showPopup) {
                        content.html(CQ.Templates.feature.viewquestion.convertCommentToAnswer({
                            title: AJS.I18n.getText("cq.question.comment.convert-to-answer.tile"),
                            body: AJS.I18n.getText("cq.question.comment.convert-to-answer.body")
                        }));

                        content.find(".cq-convert-comment-to-answer-submit").on("click", function () {
                            var questionId = $(trigger).data("question-id");
                            var commentId = $(trigger).data("comment-id");

                            //Disable confirm button
                            $(this).attr("disabled", "disabled");

                            var dfd = $.ajax({
                                type: "POST",
                                url: AJS.contextPath() + "/rest/questions/1.0/questions/" + questionId + "/comments/" + commentId + "/convert",
                                contentType: "application/json"
                            });

                            dfd.done(function (answer) {
                                analytics.trackEvent("comment", "convertedtoanswer");
                                window.location.href = AJS.contextPath() + answer.url;
                            });
                        });

                        content.find(".cq-convert-comment-to-answer-cancel").on("click", function () {
                            dialog.hide();
                        });

                        showPopup();
                    },
                    {
                        useLiveEvents: true
                    }
            );
        }

        function initAnalytics() {
            $(".cq-watch-button").on("click", function () {
                analytics.trackEvent("viewquestion", "watchbuttonclick");
            });

            $(document).on("click", ".cq-comment-question", function () {
                analytics.trackEvent("question", "addcommentclick");
            });

            $(document).on("click", ".cq-comment-answer", function () {
                analytics.trackEvent("answer", "addcommentclick");
            });
        }


        function showRelatedQuestions() {
            new SideBarQuestion().showRelatedQuestionForViewQuestion();
        }

        exports.onReady = function() {
            try {
                var questionJSON = JSON.parse(AJS.Meta.get("question"));
                var question = new Question(questionJSON, {parse: true});
                questionService.addQuestion(question);
            } catch (e) {
                console.error("Can't parse question from meta tag.", e);
            }

            comments.onReady();
            acceptAnswerLink.onReady();
            topicSuggestions.onReady();

            $(function () {
                makeDeleteDialogs();
                makeConvertToCommentDialogs();
                makeConvertToAnswerDialogs();
                showSelectedAnswer();
                focusAnswerEditor();
                initAnalytics();
                showRelatedQuestions();
            });
        };
    });
