define("cq/feature/viewquestion/comments",
    [
        "jquery",
        "underscore",
        "backbone",
        "cq/util/analytics",
        "cq/util/form",
        "cq/api/comment-collection",
        "cq/widget/editor",
        "exports"
    ],
    function (
        $,
        _,
        Backbone,
        analytics,
        formUtil,
        CommentCollection,
        Editor,
        exports) {

        var CommentView = Backbone.View.extend({
            render: function() {
                this.setElement($(CQ.Templates.feature.viewquestion.comment({ comment: this.model.attributes })));
                return this;
            }
        });

        var CommentFormView = Backbone.View.extend({
            events: {
                "click .cq-comment-submit": "submit",
                "click .cq-comment-cancel": "cancel"
            },
            initialize: function (options) {
                this.commentId = options.commentId;
                this.commentBody = options.commentBody;
            },

            render: function() {
                this.setElement($(CQ.Templates.feature.viewquestion.commentForm({
                    commentId: this.commentId,
                    commentBody: this.commentBody
                })));
                return this;
            },

            submit: function(e) {
                e.preventDefault();

                this.editor.saveContent();

                var $commentArea = this.$("#wysiwygTextarea");
                var $commentWrapper = this.$(".cq-editor-wrapper");
                if ($commentArea.val().trim().length === 0) {
                    $commentWrapper.next(".error").remove();
                    $commentWrapper.after(aui.form.fieldError({
                        message: AJS.I18n.getText("cq.question.comment.error.body.empty")
                    }));
                    return false;
                } else {
                    var dfd = this.collection.save(formUtil.serializeToJSONString(this.form()));

                    // remote error handling
                    dfd.fail(function (jqXhr) {
                        if (jqXhr.status >= 400) {
                            var error = JSON.parse(jqXhr.responseText);

                            $commentWrapper.next(".error").remove();
                            $commentWrapper.after(aui.form.fieldError({
                                message: error.errorMessage
                            }));
                        }
                    });

                    // disable buttons
                    formUtil.preventSubmission(this.form());
                    dfd.always(_.bind(function () {
                        formUtil.allowSubmission(this.form());
                    }, this));
                }
            },

            cancel: function (e) {
                if(e) {
                    e.preventDefault();
                }

                var self = this;
                this.$el.fadeOut(200, function() {
                    self.trigger("cancel");
                });
            },

            form: function() {
                return this.$(".cq-comment-form");
            },

            focus: function() {
                this.$("#wysiwygTextarea").focus();
            },
            setEditor: function(editor) {
                this.editor = editor;
            }
        });

        var CommentListView = Backbone.View.extend({
            initialize: function() {
                this.collection.on("add", this.addComment, this);
                this.collection.on("change", this.updateComment, this);
                var that = this;
                AJS.bind("remove-comment-form", function(){
                    that.removeForm();
                });
            },

            updateComment: function(comment)
            {
                var commentView = new CommentView({
                    model: comment
                });

                var $oldComment = this.$(".cq-comments-list").find('[data-content-id="' + comment.id + '"]');

                var $comment = commentView.render().$el;
                $comment.hide();
                // add highlighting
                this.$(".cq-highlighted").removeClass("cq-highlighted");
                $comment.addClass("cq-highlighted");

                $oldComment.after($comment);
                $oldComment.remove();

                this.removeForm();
                $comment.fadeIn(200);
                this.renderScriptedMacro();
            },

            addComment: function(comment) {
                var commentView = new CommentView({
                    model: comment
                });

                var $comment = commentView.render().$el;

                // add highlighting
                this.$(".cq-highlighted").removeClass("cq-highlighted");
                $comment.addClass("cq-highlighted");

                // show comment
                this.$(".cq-comments-list").append($comment.hide().fadeIn(200));
                this.removeForm();
                this.renderScriptedMacro();
            },
            showForm : function(options){
                if (!this.commentForm) {
                    this.commentForm = new CommentFormView({
                        collection: this.collection,
                        commentId: options.commentId,
                        commentBody: options.content
                    });
                    this.commentForm.on("cancel", this.removeForm, this);

                    this.$el.addClass("cq-has-content");
                    $(".cq-comment-prompt", this.$el).hide();
                    var $commentForm = this.commentForm.render().$el;
                    if (!!options.$el) {
                        options.$el.append($commentForm.fadeIn(400));;
                    } else {
                        this.$el.append($commentForm.fadeIn(400));
                    }
                }

                var editor = new Editor({el: ".cq-comment-editor-wrapper", $container: this.$el.find(".cq-editor-wrapper"), type :"COMMENT"});
                editor.createEditor();

                this.commentForm.setEditor(editor);
                this.commentForm.focus();

                this.commentForm.$el.addClass("cq-comment-form-ready");
            },

            removeForm: function() {
                if(this.commentForm) {
                    this.commentForm.remove();
                    this.commentForm = null;
                }

                $(".cq-comment-content").removeClass("hidden");

                // hacky. we should ask the collection if its empty but its not populated with server rendered comments
                var hasComments = this.$(".cq-comments-list").children().length > 0;
                if (!hasComments) {
                    this.$el.removeClass("cq-has-content");
                } else {
                    $(".cq-comment-prompt", this.$el).show();
                }
            },
            renderScriptedMacro: function () {
                //SyntaxHighlighter is a scripted macro, you need to re-call highlight
                if (window.SyntaxHighlighter) {
                    window.SyntaxHighlighter.highlight();
                }
            }
        });

        /**
         * Binds an event handler once that initialises the backbone objects, and only focuses the form afterwards.
         */
        function initComments() {
            $(document).on("click", ".cq-comment", function (e) {
                e.preventDefault();
                AJS.trigger("remove-comment-form");
                var $this = $(this);
                var $commentsContainer = $this.parents(".cq-with-comments:first");
                $('body').data("selectedCommentsContainerId", $commentsContainer.attr("id"));
                var commentListView = $commentsContainer.data("commentListView");
                if (!commentListView) {
                    var type = $this.data("type");
                    var questionId = $this.data("question-id");
                    var answerId = $this.data("answer-id");
                    var $context = $this.closest(".cq-answer, .cq-question");
                    var $comments = $(".cq-comments", $context);

                    var commentCollection = new CommentCollection([], {
                        type: type,
                        questionId: questionId,
                        answerId: answerId
                    });

                    var commentListView = new CommentListView({
                        el: $comments,
                        collection: commentCollection
                    });
                    $commentsContainer.data("commentListView", commentListView);
                }

                // we must wait until the answer comment is fully loaded
                if (AJS.CQ.EditorLoader.resourcesLoaded()) {
                    commentListView.showForm({$el: null, content: "", commentId : null});
                }
            });

            $(document).on("click", ".cq-edit-comment", function (e) {
                e.preventDefault();
                AJS.trigger("remove-comment-form");
                var $this = $(this);
                var $commentsContainer = $this.parents(".cq-with-comments:first");
                $('body').data("selectedCommentsContainerId", $commentsContainer.attr("id"));
                var $commentContainer = $this.parents(".cq-comment-container:first");
                var $editorHolder =  $commentContainer.find(".edit-comment-editor-holder");
                var commentContent =  $commentContainer.find(".cq-comment-content-raw").val();
                var commentId =  $this.data("comment-id");

                $this.parents(".cq-comment-container").find(".cq-comment-content").addClass("hidden");

                var type = $this.data("type");
                var questionId = $this.data("question-id");
                var answerId = $this.data("answer-id");
                var $context = $this.closest(".cq-answer, .cq-question");
                var $comments = $(".cq-comments", $context);

                var comment = {id: commentId};

                var commentCollection = new CommentCollection([comment], {
                    type: type,
                    questionId: questionId,
                    answerId: answerId
                });

                var commentListView = new CommentListView({
                    el: $comments,
                    collection: commentCollection
                });
                $commentsContainer.data("commentListView", commentListView);

                if (AJS.CQ.EditorLoader.resourcesLoaded()) {
                    commentListView.showForm({$el: $editorHolder, content: commentContent, commentId : commentId});
                }
            });
        }

        exports.onReady = function() {
            $(function() {
                console.log("**** comments.js ==> onReady! Calling initComments to bind click events!!!");
                initComments();
            })
        };
    });

require(["jquery", "ajs"], function ($, AJS) {
    var showFormWhenEditorReady = function() {
        var $commentsContainer = $("#" + $('body').data("selectedCommentsContainerId"));
        if (!!$commentsContainer) {
            var commentListView = $commentsContainer.data("commentListView");
            if (!!commentListView) {
                if (commentListView.collection.length  > 0) {
                    var commentId =  commentListView.collection.at(0).id;
                    var $commentContainer = $commentsContainer.find('[data-content-id="' + commentId + '"] .cq-comment-container').first();
                    var $editorHolder = $commentContainer.find(".edit-comment-editor-holder");
                    var commentContent = $commentContainer.find(".cq-comment-content-raw").val();
                    commentListView.showForm({$el: $editorHolder, content: commentContent, commentId : commentId});
                }
                else {
                    commentListView.showForm({$el: null, content: "", commentId : null});
                }
            }
        }
        AJS.unbind("cq-editor-ready", showFormWhenEditorReady);
    };
    AJS.bind("cq-editor-ready", showFormWhenEditorReady);
});
