define("cq/feature/stats",
    [
        "jquery",
        "underscore",
        "cq/util/analytics",
        "cq/widget/loading",
        "cq/widget/tooltips",
        "cq/widget/spacepicker",
        "exports"
    ],
    function (
        $,
        _,
        analytics,
        loading,
        tooltips,
        SpacePicker,
        exports) {

        // Random sample data created via PopulateQuestionsTest
        var BLANK_EXP_DATA = '{"com.atlassian.confluence.plugins.confluence-questions:question":[{"date":1388322000000,"value":4,"bucket":-13},{"date":1388926800000,"value":1,"bucket":-12},{"date":1389531600000,"value":4,"bucket":-11},{"date":1390136400000,"value":2,"bucket":-10},{"date":1390741200000,"value":3,"bucket":-9},{"date":1391346000000,"value":3,"bucket":-8},{"date":1391950800000,"value":3,"bucket":-7},{"date":1392555600000,"value":4,"bucket":-6},{"date":1393160400000,"value":2,"bucket":-5},{"date":1393765200000,"value":3,"bucket":-4},{"date":1394370000000,"value":3,"bucket":-3},{"date":1394974800000,"value":1,"bucket":-2},{"date":1395579600000,"value":2,"bucket":-1}],"com.atlassian.confluence.plugins.confluence-questions:comment":[{"date":1388322000000,"value":0,"bucket":-13},{"date":1388926800000,"value":0,"bucket":-12},{"date":1389531600000,"value":1,"bucket":-11},{"date":1390136400000,"value":0,"bucket":-10},{"date":1390741200000,"value":0,"bucket":-9},{"date":1391346000000,"value":0,"bucket":-8},{"date":1391950800000,"value":1,"bucket":-7},{"date":1392555600000,"value":4,"bucket":-6},{"date":1393160400000,"value":4,"bucket":-5},{"date":1393765200000,"value":7,"bucket":-4},{"date":1394370000000,"value":8,"bucket":-3},{"date":1394974800000,"value":9,"bucket":-2},{"date":1395579600000,"value":14,"bucket":-1}],"com.atlassian.confluence.plugins.confluence-questions:answer":[{"date":1388322000000,"value":0,"bucket":-13},{"date":1388926800000,"value":0,"bucket":-12},{"date":1389531600000,"value":1,"bucket":-11},{"date":1390136400000,"value":1,"bucket":-10},{"date":1390741200000,"value":1,"bucket":-9},{"date":1391346000000,"value":2,"bucket":-8},{"date":1391950800000,"value":2,"bucket":-7},{"date":1392555600000,"value":2,"bucket":-6},{"date":1393160400000,"value":3,"bucket":-5},{"date":1393765200000,"value":2,"bucket":-4},{"date":1394370000000,"value":4,"bucket":-3},{"date":1394974800000,"value":8,"bucket":-2},{"date":1395579600000,"value":3,"bucket":-1}]}';

        function getMinMaxValues(charts) {
            var yMin = 0; // Always use 0 as baseline
            var yMax = Number.MIN_VALUE;
            var xMin = Number.MAX_VALUE;
            var xMax = Number.MIN_VALUE;

            _.each(charts, function (chart) {
                _.each(chart.data, function (data) {
                    yMax = Math.max(yMax, data.value);
                    xMin = Math.min(xMin, data.date);
                    xMax = Math.max(xMax, data.date);
                });
            });

            return [
                [xMin, yMin],
                [xMax, yMax]
            ];
        }

        function chartsFromData(data) {
            var dataPointToArray = function(datapoint) { return [datapoint.date, datapoint.value] };
            var chartValues = function () { return _.map(this.data, dataPointToArray); };

            var questionsChart = {
                data: data["com.atlassian.confluence.plugins.confluence-questions:question"],
                color: "red",
                label: AJS.I18n.getText("cq.stats.chart.questions.per.week.label"),
                values: chartValues
            };

            var answersChart = {
                data:  data["com.atlassian.confluence.plugins.confluence-questions:answer"],
                color: "green",
                label: AJS.I18n.getText("cq.stats.chart.answers.per.week.label"),
                values: chartValues
            };

            var commentsChart = {
                data: data["com.atlassian.confluence.plugins.confluence-questions:comment"],
                color: "blue",
                label: AJS.I18n.getText("cq.stats.chart.comments.per.week.label"),
                values: chartValues
            };

            var charts = [questionsChart, answersChart, commentsChart];

            // remove most recent bucket in charts because it contains the current week which might not be finished
            // yet and would always show a drop off in stats
            charts = _.map(charts, function (chart) {
                if (_.last(chart.data).bucket === 0) {
                    chart.data.pop();
                }

                return chart;
            });

            // only show charts that have at least 2 data points
            charts = _.filter(charts, function(chart) {
                return chart.data.length >= 2;
            });

            return charts;
        }

        function renderChart(data) {
            var dataPointDate = function(datapoint) { return datapoint.date; };

            function linePlot() {
                return c3.linePlot().lineConstructor(function () {
                    return d3.svg.line().interpolate("monotone");
                });
            }

            var charts = chartsFromData(data);
            var blankExperience = false;

            if (charts.length == 0) {
                // blank exp, get dummy data
                charts = chartsFromData(JSON.parse(BLANK_EXP_DATA));
                blankExperience = true;
            }


            var plots = c3.layerable();

            // show grid
            plots.addLayer("grid", c3.axis()
                .axisConstructor(function() {
                    var tickSize = -this.width();
                    return d3.svg.axis().tickSize(tickSize).ticks(4).tickFormat('');
                }).orient("left"));

            _.each(charts, function (chart) {
                plots.addLayer(chart.color, linePlot().data(chart.values()))
            });

            var chartFn = c3.component("cqChart")
                .extend(c3.borderLayout())
                .south(c3.axis()
                .axisConstructor(function () {
                    return d3.svg.axis()
                        .tickFormat(function(d) {
                            return d3.time.format("%d/%m")(new Date(d));
                        })
                        .tickValues(_.map(_.first(charts).data, dataPointDate));
                })
                .height(30))
                .west(c3.axis()
                    .axisConstructor(function () {
                        return d3.svg.axis()
                            .tickSize(0)
                            .tickPadding(7)
                            .ticks(4);
                    })
                    .orient("left").width(30))
                .north(c3.withDimensions().height(10)) // just some padding
                .east(c3.withDimensions().width(20)) // just some padding
                .center(plots)
                .data(getMinMaxValues(charts));

            chartFn(d3.select(".cq-stats-charts svg"));
            var $chartOuter = $(".cq-stats-charts");

            if (blankExperience) {
                $chartOuter.addClass("cq-chart-blank-experience");
                $chartOuter.append(CQ.Templates.feature.stats.blankMessage());
            }

            if (!blankExperience) {
                var $blankMessage = $(".cq-blank-message");
                $chartOuter.removeClass("cq-chart-blank-experience");
                $blankMessage.remove();
                renderLegend(charts);
            }
        }

        function renderLegend(charts) {
            $(".cq-stats-charts-legend").html(CQ.Templates.feature.stats.chartLegend({charts: charts}));
        }

        function renderStats(stats) {
            $(".cq-stats-summary").html(CQ.Templates.feature.stats.statsSummary({stats: stats}));
            tooltips.initialize(".cq-stats-summary");
        }

        function getStats(spaceKey) {
            var suffix = "";

            if (spaceKey != "") {
                suffix = "/space/" + spaceKey;
            }

            return  $.ajax({
                url: AJS.contextPath() + "/rest/questions/1.0/statistics" + suffix,
                cache: false,
                type: "GET",
                dataType: "json",
                data: {
                    statistics: ["questions_per_day", "answered_ratio", "total_contributors", "answers_ratio"]
                }
            });
        }

        function getChartData(spaceKey) {
            var suffix = "/timeseries";

            if (spaceKey != "") {
                suffix = "/space/" + spaceKey + suffix;
            }

            return  $.ajax({
                url: AJS.contextPath() + "/rest/questions/1.0/statistics" + suffix,
                cache: false,
                type: "GET",
                dataType: "json"
            });
        }

        function drawData(spaceKey) {
            var statsDfd = getStats(spaceKey);
            var chartDfd = getChartData(spaceKey);
            var masterDfd = $.when(statsDfd, chartDfd);
            var $container = $(".cq-stats-data");

            statsDfd.done(function (data) {
                renderStats(data);
            });

            chartDfd.done(function (data) {
                renderChart(data);
            });

            // fade in
            $container.css({opacity: 0});
            masterDfd.done(function () {
                $container.animate({opacity: 1}, 500);
            });

            // ajax indicator for tests
            $container.attr("data-ajax-request", true);
            masterDfd.always(function () {
                $container.attr("data-ajax-request", false);
            });

            // spinner
            var sectionSpinner = new loading.SectionSpinner($(".cq-stats"));
            sectionSpinner.showUntilResolved(masterDfd);
        };

        var $target = $("#spaceKey");
        var spacePicker = new SpacePicker($target, AJS.I18n.getText("cq.question.stats.spacekey.placeholder"));

        spacePicker.on("change", function(spaceKey) {
            drawData(spaceKey);
        });

        var defaultSpaceKey = $target.attr("value");
        if (defaultSpaceKey) {
            drawData(defaultSpaceKey);
        } else {
            drawData("");
        }

        exports.onReady = function() { };
    });