/*
 * Decompiled with CFR 0.152.
 */
package org.apache.shindig.gadgets.oauth;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Pattern;
import net.oauth.OAuth;
import net.oauth.OAuthAccessor;
import net.oauth.OAuthException;
import net.oauth.OAuthMessage;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.shindig.auth.OAuthUtil;
import org.apache.shindig.common.uri.Uri;
import org.apache.shindig.common.uri.UriBuilder;
import org.apache.shindig.common.util.CharsetUtil;
import org.apache.shindig.gadgets.ChainedContentFetcher;
import org.apache.shindig.gadgets.GadgetException;
import org.apache.shindig.gadgets.http.HttpFetcher;
import org.apache.shindig.gadgets.http.HttpRequest;
import org.apache.shindig.gadgets.http.HttpResponse;
import org.apache.shindig.gadgets.http.HttpResponseBuilder;
import org.apache.shindig.gadgets.oauth.AccessorInfo;
import org.apache.shindig.gadgets.oauth.OAuthClientState;
import org.apache.shindig.gadgets.oauth.OAuthError;
import org.apache.shindig.gadgets.oauth.OAuthFetcherConfig;
import org.apache.shindig.gadgets.oauth.OAuthProtocolException;
import org.apache.shindig.gadgets.oauth.OAuthResponseParams;
import org.apache.shindig.gadgets.oauth.OAuthStore;
import org.json.JSONObject;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class OAuthFetcher
extends ChainedContentFetcher {
    private static final int MAX_ATTEMPTS = 2;
    public static final String XOAUTH_APP_URL = "xoauth_app_url";
    protected static final String OPENSOCIAL_OWNERID = "opensocial_owner_id";
    protected static final String OPENSOCIAL_VIEWERID = "opensocial_viewer_id";
    protected static final String OPENSOCIAL_APPID = "opensocial_app_id";
    protected static final String OPENSOCIAL_APPURL = "opensocial_app_url";
    protected static final String OPENSOCIAL_PROXIED_CONTENT = "opensocial_proxied_content";
    protected static final String XOAUTH_PUBLIC_KEY = "xoauth_signature_publickey";
    protected static final Pattern ALLOWED_PARAM_NAME = Pattern.compile("[-:\\w~!@$*()_\\[\\]:,./]+");
    private static final long ACCESS_TOKEN_EXPIRE_UNKNOWN = 0L;
    private static final long ACCESS_TOKEN_FORCE_EXPIRE = -1L;
    private static final AtomicLong atomicNonce = new AtomicLong();
    protected final OAuthFetcherConfig fetcherConfig;
    private final List<OAuth.Parameter> trustedParams;
    protected OAuthClientState clientState;
    protected OAuthResponseParams responseParams;
    private AccessorInfo accessorInfo;
    private HttpRequest realRequest;
    private Map<String, String> accessTokenData;

    public OAuthFetcher(OAuthFetcherConfig fetcherConfig, HttpFetcher fetcher) {
        this(fetcherConfig, fetcher, null);
    }

    public OAuthFetcher(OAuthFetcherConfig fetcherConfig, HttpFetcher fetcher, List<OAuth.Parameter> trustedParams) {
        super(fetcher);
        this.fetcherConfig = fetcherConfig;
        this.trustedParams = trustedParams;
    }

    @Override
    public HttpResponse fetch(HttpRequest request) {
        this.realRequest = request;
        this.clientState = new OAuthClientState(this.fetcherConfig.getStateCrypter(), request.getOAuthArguments().getOrigClientState());
        this.responseParams = new OAuthResponseParams(request.getSecurityToken(), request, this.fetcherConfig.getStateCrypter());
        try {
            return this.fetchNoThrow();
        }
        catch (RuntimeException e) {
            this.responseParams.logDetailedWarning("OAuth fetch unexpected fatal error", e);
            throw e;
        }
    }

    private HttpResponse fetchNoThrow() {
        HttpResponseBuilder response = null;
        try {
            this.accessorInfo = this.fetcherConfig.getTokenStore().getOAuthAccessor(this.realRequest.getSecurityToken(), this.realRequest.getOAuthArguments(), this.clientState, this.responseParams);
            response = this.fetchWithRetry();
        }
        catch (OAuthResponseParams.OAuthRequestException e) {
            if (OAuthError.UNAUTHENTICATED.toString().equals(this.responseParams.getError())) {
                this.responseParams.logDetailedInfo("Unauthenticated OAuth fetch", e);
            } else {
                this.responseParams.logDetailedWarning("OAuth fetch fatal error", e);
            }
            this.responseParams.setSendTraceToClient(true);
            response = new HttpResponseBuilder().setHttpStatusCode(403).setStrictNoCache();
            this.responseParams.addToResponse(response);
            return response.create();
        }
        if (response.getHttpStatusCode() >= 400) {
            this.responseParams.logDetailedWarning("OAuth fetch fatal error");
            this.responseParams.setSendTraceToClient(true);
        } else if (this.responseParams.getAznUrl() != null && this.responseParams.sawErrorResponse()) {
            this.responseParams.logDetailedWarning("OAuth fetch error, reprompting for user approval");
            this.responseParams.setSendTraceToClient(true);
        }
        this.responseParams.addToResponse(response);
        return response.create();
    }

    private HttpResponseBuilder fetchWithRetry() throws OAuthResponseParams.OAuthRequestException {
        boolean retry;
        int attempts = 0;
        HttpResponseBuilder response = null;
        do {
            retry = false;
            ++attempts;
            try {
                response = this.attemptFetch();
            }
            catch (OAuthProtocolException pe) {
                retry = this.handleProtocolException(pe, attempts);
                if (retry) continue;
                if (pe.getProblemCode() != null) {
                    throw this.responseParams.oauthRequestException(pe.getProblemCode(), "Service provider rejected request", (Throwable)pe);
                }
                throw this.responseParams.oauthRequestException(OAuthError.UNKNOWN_PROBLEM, "Service provider rejected request", (Throwable)pe);
            }
        } while (retry);
        return response;
    }

    private boolean handleProtocolException(OAuthProtocolException pe, int attempts) throws OAuthResponseParams.OAuthRequestException {
        if (pe.canExtend()) {
            this.accessorInfo.setTokenExpireMillis(-1L);
        } else if (pe.startFromScratch()) {
            this.fetcherConfig.getTokenStore().removeToken(this.realRequest.getSecurityToken(), this.accessorInfo.getConsumer(), this.realRequest.getOAuthArguments(), this.responseParams);
            this.accessorInfo.getAccessor().accessToken = null;
            this.accessorInfo.getAccessor().requestToken = null;
            this.accessorInfo.getAccessor().tokenSecret = null;
            this.accessorInfo.setSessionHandle(null);
            this.accessorInfo.setTokenExpireMillis(0L);
        }
        return attempts < 2 && pe.canRetry();
    }

    private HttpResponseBuilder attemptFetch() throws OAuthResponseParams.OAuthRequestException, OAuthProtocolException {
        if (this.needApproval()) {
            this.checkCanApprove();
            this.fetchRequestToken();
            this.buildClientApprovalState();
            this.buildAznUrl();
            return new HttpResponseBuilder().setHttpStatusCode(200).setStrictNoCache();
        }
        if (this.needAccessToken()) {
            this.checkCanApprove();
            this.exchangeRequestToken();
            this.saveAccessToken();
            this.buildClientAccessState();
        }
        return this.fetchData();
    }

    private boolean needApproval() {
        return this.realRequest.getOAuthArguments().mustUseToken() && this.accessorInfo.getAccessor().requestToken == null && this.accessorInfo.getAccessor().accessToken == null;
    }

    private void checkCanApprove() throws OAuthResponseParams.OAuthRequestException {
        String pageOwner = this.realRequest.getSecurityToken().getOwnerId();
        String pageViewer = this.realRequest.getSecurityToken().getViewerId();
        String stateOwner = this.clientState.getOwner();
        if (pageOwner == null) {
            throw this.responseParams.oauthRequestException(OAuthError.UNAUTHENTICATED, "Unauthenticated");
        }
        if (!pageOwner.equals(pageViewer)) {
            throw this.responseParams.oauthRequestException(OAuthError.NOT_OWNER, "Only page owners can grant OAuth approval");
        }
        if (stateOwner != null && !stateOwner.equals(pageOwner)) {
            throw this.responseParams.oauthRequestException(OAuthError.UNKNOWN_PROBLEM, "Client state belongs to a different person (state owner=" + stateOwner + ", pageOwner=" + pageOwner + ')');
        }
    }

    private void fetchRequestToken() throws OAuthResponseParams.OAuthRequestException, OAuthProtocolException {
        OAuthAccessor accessor = this.accessorInfo.getAccessor();
        HttpRequest request = this.createRequestTokenRequest(accessor);
        ArrayList requestTokenParams = Lists.newArrayList();
        this.addCallback(requestTokenParams);
        HttpRequest signed = this.sanitizeAndSign(request, requestTokenParams, true);
        OAuthMessage reply = this.sendOAuthMessage(signed);
        accessor.requestToken = OAuthUtil.getParameter((OAuthMessage)reply, (String)"oauth_token");
        accessor.tokenSecret = OAuthUtil.getParameter((OAuthMessage)reply, (String)"oauth_token_secret");
    }

    private HttpRequest createRequestTokenRequest(OAuthAccessor accessor) throws OAuthResponseParams.OAuthRequestException {
        if (accessor.consumer.serviceProvider.requestTokenURL == null) {
            throw this.responseParams.oauthRequestException(OAuthError.BAD_OAUTH_CONFIGURATION, "No request token URL specified");
        }
        HttpRequest request = new HttpRequest(Uri.parse((String)accessor.consumer.serviceProvider.requestTokenURL));
        request.setMethod(this.accessorInfo.getHttpMethod().toString());
        if (this.accessorInfo.getHttpMethod() == AccessorInfo.HttpMethod.POST) {
            request.setHeader("Content-Type", "application/x-www-form-urlencoded");
        }
        return request;
    }

    private void addCallback(List<OAuth.Parameter> requestTokenParams) throws OAuthResponseParams.OAuthRequestException {
        String callbackUrl;
        String baseCallback = StringUtils.trimToNull((String)this.accessorInfo.getConsumer().getCallbackUrl());
        if (baseCallback != null && (callbackUrl = this.fetcherConfig.getOAuthCallbackGenerator().generateCallback(this.fetcherConfig, baseCallback, this.realRequest, this.responseParams)) != null) {
            requestTokenParams.add(new OAuth.Parameter("oauth_callback", callbackUrl));
        }
    }

    private List<OAuth.Parameter> sanitize(List<OAuth.Parameter> params) throws OAuthResponseParams.OAuthRequestException {
        ArrayList list = Lists.newArrayList();
        for (OAuth.Parameter p : params) {
            String name = p.getKey();
            if (this.allowParam(name)) {
                list.add(p);
                continue;
            }
            throw this.responseParams.oauthRequestException(OAuthError.INVALID_REQUEST, "invalid parameter name " + name + ", applications may not override opensocial or oauth parameters");
        }
        return list;
    }

    private boolean allowParam(String paramName) {
        String canonParamName = paramName.toLowerCase();
        return !canonParamName.startsWith("oauth") && !canonParamName.startsWith("xoauth") && !canonParamName.startsWith("opensocial") && ALLOWED_PARAM_NAME.matcher(canonParamName).matches();
    }

    private void addIdentityParams(List<OAuth.Parameter> params) {
        String appUrl;
        String app;
        String viewer;
        if (!this.realRequest.getOAuthArguments().getSignOwner() && !this.realRequest.getOAuthArguments().getSignViewer()) {
            return;
        }
        String owner = this.realRequest.getSecurityToken().getOwnerId();
        if (owner != null && this.realRequest.getOAuthArguments().getSignOwner()) {
            params.add(new OAuth.Parameter(OPENSOCIAL_OWNERID, owner));
        }
        if ((viewer = this.realRequest.getSecurityToken().getViewerId()) != null && this.realRequest.getOAuthArguments().getSignViewer()) {
            params.add(new OAuth.Parameter(OPENSOCIAL_VIEWERID, viewer));
        }
        if ((app = this.realRequest.getSecurityToken().getAppId()) != null) {
            params.add(new OAuth.Parameter(OPENSOCIAL_APPID, app));
        }
        if ((appUrl = this.realRequest.getSecurityToken().getAppUrl()) != null) {
            params.add(new OAuth.Parameter(OPENSOCIAL_APPURL, appUrl));
        }
        if (this.trustedParams != null) {
            params.addAll(this.trustedParams);
        }
        if (this.realRequest.getOAuthArguments().isProxiedContentRequest()) {
            params.add(new OAuth.Parameter(OPENSOCIAL_PROXIED_CONTENT, "1"));
        }
    }

    private void addSignatureParams(List<OAuth.Parameter> params) {
        if (this.accessorInfo.getConsumer().getConsumer().consumerKey == null) {
            params.add(new OAuth.Parameter("oauth_consumer_key", this.realRequest.getSecurityToken().getDomain()));
        }
        if (this.accessorInfo.getConsumer().getKeyName() != null) {
            params.add(new OAuth.Parameter(XOAUTH_PUBLIC_KEY, this.accessorInfo.getConsumer().getKeyName()));
        }
        params.add(new OAuth.Parameter("oauth_version", "1.0"));
        params.add(new OAuth.Parameter("oauth_timestamp", Long.toString(this.fetcherConfig.getClock().currentTimeMillis() / 1000L)));
        params.add(new OAuth.Parameter("oauth_nonce", Long.toString(atomicNonce.incrementAndGet())));
    }

    static String getAuthorizationHeader(List<Map.Entry<String, String>> oauthParams) {
        StringBuilder result = new StringBuilder("OAuth ");
        boolean first = true;
        for (Map.Entry<String, String> parameter : oauthParams) {
            if (!first) {
                result.append(", ");
            } else {
                first = false;
            }
            result.append(OAuth.percentEncode((String)parameter.getKey())).append("=\"").append(OAuth.percentEncode((String)parameter.getValue())).append('\"');
        }
        return result.toString();
    }

    public HttpRequest sanitizeAndSign(HttpRequest base, List<OAuth.Parameter> params, boolean tokenEndpoint) throws OAuthResponseParams.OAuthRequestException {
        if (params == null) {
            params = Lists.newArrayList();
        }
        UriBuilder target = new UriBuilder(base.getUri());
        String query = target.getQuery();
        target.setQuery(null);
        params.addAll(this.sanitize(OAuth.decodeForm((String)query)));
        switch (OAuthUtil.getSignatureType((boolean)tokenEndpoint, (String)base.getHeader("Content-Type"))) {
            case URL_ONLY: {
                break;
            }
            case URL_AND_FORM_PARAMS: {
                params.addAll(this.sanitize(OAuth.decodeForm((String)base.getPostBodyAsString())));
                break;
            }
            case URL_AND_BODY_HASH: {
                try {
                    byte[] body = IOUtils.toByteArray((InputStream)base.getPostBody());
                    byte[] hash = DigestUtils.sha((byte[])body);
                    String b64 = new String(Base64.encodeBase64((byte[])hash), CharsetUtil.UTF8.name());
                    params.add(new OAuth.Parameter("oauth_body_hash", b64));
                    break;
                }
                catch (IOException e) {
                    throw this.responseParams.oauthRequestException(OAuthError.UNKNOWN_PROBLEM, "Error taking body hash", (Throwable)e);
                }
            }
        }
        this.addIdentityParams(params);
        this.addSignatureParams(params);
        try {
            OAuthMessage signed = this.sign(this.accessorInfo.getAccessor(), base.getMethod(), target.toString(), params);
            HttpRequest oauthHttpRequest = this.createHttpRequest(base, OAuthFetcher.selectOAuthParams(signed));
            oauthHttpRequest.setFollowRedirects(false);
            oauthHttpRequest.setIgnoreCache(true);
            return oauthHttpRequest;
        }
        catch (OAuthException e) {
            throw this.responseParams.oauthRequestException(OAuthError.UNKNOWN_PROBLEM, "Error signing message", (Throwable)e);
        }
    }

    protected OAuthMessage sign(OAuthAccessor accessor, String httpMethod, String uri, List<OAuth.Parameter> params) throws OAuthException {
        return OAuthUtil.newRequestMessage((OAuthAccessor)accessor, (String)httpMethod, (String)uri, params);
    }

    private HttpRequest createHttpRequest(HttpRequest base, List<Map.Entry<String, String>> oauthParams) throws OAuthResponseParams.OAuthRequestException {
        AccessorInfo.OAuthParamLocation paramLocation = this.accessorInfo.getParamLocation();
        HttpRequest result = new HttpRequest(base);
        if (paramLocation == AccessorInfo.OAuthParamLocation.POST_BODY && !result.getMethod().equals("POST")) {
            paramLocation = AccessorInfo.OAuthParamLocation.AUTH_HEADER;
        }
        switch (paramLocation) {
            case AUTH_HEADER: {
                result.addHeader("Authorization", OAuthFetcher.getAuthorizationHeader(oauthParams));
                break;
            }
            case POST_BODY: {
                String contentType = result.getHeader("Content-Type");
                if (!OAuth.isFormEncoded((String)contentType)) {
                    throw this.responseParams.oauthRequestException(OAuthError.INVALID_REQUEST, "OAuth param location can only be post_body if post body if of type x-www-form-urlencoded");
                }
                String oauthData = OAuthUtil.formEncode(oauthParams);
                if (result.getPostBodyLength() == 0) {
                    result.setPostBody(CharsetUtil.getUtf8Bytes((String)oauthData));
                    break;
                }
                result.setPostBody((result.getPostBodyAsString() + '&' + oauthData).getBytes());
                break;
            }
            case URI_QUERY: {
                result.setUri(Uri.parse((String)OAuthUtil.addParameters((String)result.getUri().toString(), oauthParams)));
            }
        }
        return result;
    }

    private OAuthMessage sendOAuthMessage(HttpRequest request) throws OAuthResponseParams.OAuthRequestException, OAuthProtocolException {
        HttpResponse response = this.fetchFromServer(request);
        this.checkForProtocolProblem(response);
        OAuthMessage reply = new OAuthMessage(null, null, null);
        reply.addParameters((Collection)OAuth.decodeForm((String)response.getResponseAsString()));
        reply = this.parseAuthHeader(reply, response);
        if (OAuthUtil.getParameter((OAuthMessage)reply, (String)"oauth_token") == null) {
            throw this.responseParams.oauthRequestException(OAuthError.UNKNOWN_PROBLEM, "No oauth_token returned from service provider");
        }
        if (OAuthUtil.getParameter((OAuthMessage)reply, (String)"oauth_token_secret") == null) {
            throw this.responseParams.oauthRequestException(OAuthError.UNKNOWN_PROBLEM, "No oauth_token_secret returned from service provider");
        }
        return reply;
    }

    private OAuthMessage parseAuthHeader(OAuthMessage msg, HttpResponse resp) {
        if (msg == null) {
            msg = new OAuthMessage(null, null, null);
        }
        for (String auth : resp.getHeaders("WWW-Authenticate")) {
            msg.addParameters((Collection)OAuthMessage.decodeAuthorization((String)auth));
        }
        return msg;
    }

    private void buildClientApprovalState() {
        OAuthAccessor accessor = this.accessorInfo.getAccessor();
        this.responseParams.getNewClientState().setRequestToken(accessor.requestToken);
        this.responseParams.getNewClientState().setRequestTokenSecret(accessor.tokenSecret);
        this.responseParams.getNewClientState().setOwner(this.realRequest.getSecurityToken().getOwnerId());
    }

    private void buildAznUrl() throws OAuthResponseParams.OAuthRequestException {
        OAuthAccessor accessor = this.accessorInfo.getAccessor();
        if (accessor.consumer.serviceProvider.userAuthorizationURL == null) {
            throw this.responseParams.oauthRequestException(OAuthError.BAD_OAUTH_CONFIGURATION, "No authorization URL specified");
        }
        StringBuilder azn = new StringBuilder(accessor.consumer.serviceProvider.userAuthorizationURL);
        if (azn.indexOf("?") == -1) {
            azn.append('?');
        } else {
            azn.append('&');
        }
        azn.append("oauth_token");
        azn.append('=');
        azn.append(OAuth.percentEncode((String)accessor.requestToken));
        this.responseParams.setAznUrl(azn.toString());
    }

    private boolean needAccessToken() {
        if (this.realRequest.getOAuthArguments().mustUseToken() && this.accessorInfo.getAccessor().requestToken != null && this.accessorInfo.getAccessor().accessToken == null) {
            return true;
        }
        return this.realRequest.getOAuthArguments().mayUseToken() && this.accessTokenExpired();
    }

    private boolean accessTokenExpired() {
        return this.accessorInfo.getTokenExpireMillis() != 0L && this.accessorInfo.getTokenExpireMillis() < this.fetcherConfig.getClock().currentTimeMillis();
    }

    private void exchangeRequestToken() throws OAuthResponseParams.OAuthRequestException, OAuthProtocolException {
        String receivedCallback;
        if (this.accessorInfo.getAccessor().accessToken != null) {
            this.accessorInfo.getAccessor().requestToken = this.accessorInfo.getAccessor().accessToken;
            this.accessorInfo.getAccessor().accessToken = null;
        }
        OAuthAccessor accessor = this.accessorInfo.getAccessor();
        if (accessor.consumer.serviceProvider.accessTokenURL == null) {
            throw this.responseParams.oauthRequestException(OAuthError.BAD_OAUTH_CONFIGURATION, "No access token URL specified.");
        }
        Uri accessTokenUri = Uri.parse((String)accessor.consumer.serviceProvider.accessTokenURL);
        HttpRequest request = new HttpRequest(accessTokenUri);
        request.setMethod(this.accessorInfo.getHttpMethod().toString());
        if (this.accessorInfo.getHttpMethod() == AccessorInfo.HttpMethod.POST) {
            request.setHeader("Content-Type", "application/x-www-form-urlencoded");
        }
        ArrayList msgParams = Lists.newArrayList();
        msgParams.add(new OAuth.Parameter("oauth_token", accessor.requestToken));
        if (this.accessorInfo.getSessionHandle() != null) {
            msgParams.add(new OAuth.Parameter("oauth_session_handle", this.accessorInfo.getSessionHandle()));
        }
        if (!StringUtils.isBlank((String)(receivedCallback = this.realRequest.getOAuthArguments().getReceivedCallbackUrl()))) {
            try {
                Uri parsed = Uri.parse((String)receivedCallback);
                String verifier = parsed.getQueryParameter("oauth_verifier");
                if (verifier != null) {
                    msgParams.add(new OAuth.Parameter("oauth_verifier", verifier));
                }
            }
            catch (IllegalArgumentException e) {
                throw this.responseParams.oauthRequestException(OAuthError.INVALID_REQUEST, "Invalid received callback URL: " + receivedCallback, (Throwable)e);
            }
        }
        HttpRequest signed = this.sanitizeAndSign(request, msgParams, true);
        OAuthMessage reply = this.sendOAuthMessage(signed);
        accessor.accessToken = OAuthUtil.getParameter((OAuthMessage)reply, (String)"oauth_token");
        accessor.tokenSecret = OAuthUtil.getParameter((OAuthMessage)reply, (String)"oauth_token_secret");
        this.accessorInfo.setSessionHandle(OAuthUtil.getParameter((OAuthMessage)reply, (String)"oauth_session_handle"));
        this.accessorInfo.setTokenExpireMillis(0L);
        if (OAuthUtil.getParameter((OAuthMessage)reply, (String)"oauth_expires_in") != null) {
            try {
                long expireSecs = Long.parseLong(OAuthUtil.getParameter((OAuthMessage)reply, (String)"oauth_expires_in"));
                long expireMillis = this.fetcherConfig.getClock().currentTimeMillis() + expireSecs * 1000L;
                this.accessorInfo.setTokenExpireMillis(expireMillis);
            }
            catch (NumberFormatException e) {
                this.responseParams.logDetailedWarning("server returned bogus expiration");
            }
        }
        if (accessTokenUri.equals((Object)this.realRequest.getUri())) {
            this.accessTokenData = Maps.newHashMap();
            for (Map.Entry param : OAuthUtil.getParameters((OAuthMessage)reply)) {
                if (((String)param.getKey()).startsWith("oauth")) continue;
                this.accessTokenData.put((String)param.getKey(), (String)param.getValue());
            }
        }
    }

    private void saveAccessToken() throws OAuthResponseParams.OAuthRequestException {
        OAuthAccessor accessor = this.accessorInfo.getAccessor();
        OAuthStore.TokenInfo tokenInfo = new OAuthStore.TokenInfo(accessor.accessToken, accessor.tokenSecret, this.accessorInfo.getSessionHandle(), this.accessorInfo.getTokenExpireMillis());
        this.fetcherConfig.getTokenStore().storeTokenKeyAndSecret(this.realRequest.getSecurityToken(), this.accessorInfo.getConsumer(), this.realRequest.getOAuthArguments(), tokenInfo, this.responseParams);
    }

    private void buildClientAccessState() {
        OAuthAccessor accessor = this.accessorInfo.getAccessor();
        this.responseParams.getNewClientState().setAccessToken(accessor.accessToken);
        this.responseParams.getNewClientState().setAccessTokenSecret(accessor.tokenSecret);
        this.responseParams.getNewClientState().setOwner(this.realRequest.getSecurityToken().getOwnerId());
        this.responseParams.getNewClientState().setSessionHandle(this.accessorInfo.getSessionHandle());
        this.responseParams.getNewClientState().setTokenExpireMillis(this.accessorInfo.getTokenExpireMillis());
    }

    private HttpResponseBuilder fetchData() throws OAuthResponseParams.OAuthRequestException, OAuthProtocolException {
        HttpResponseBuilder builder = null;
        if (this.accessTokenData != null) {
            builder = this.formatAccessTokenData();
        } else {
            HttpRequest signed = this.sanitizeAndSign(this.realRequest, null, false);
            HttpResponse response = this.fetchFromServer(signed);
            this.checkForProtocolProblem(response);
            builder = new HttpResponseBuilder(response);
        }
        return builder;
    }

    private HttpResponse fetchFromServer(HttpRequest request) throws OAuthResponseParams.OAuthRequestException {
        HttpResponse response = null;
        try {
            response = this.nextFetcher.fetch(request);
            if (response == null) {
                throw this.responseParams.oauthRequestException(OAuthError.UNKNOWN_PROBLEM, "No response from server");
            }
            HttpResponse httpResponse = response;
            return httpResponse;
        }
        catch (GadgetException e) {
            throw this.responseParams.oauthRequestException(OAuthError.UNKNOWN_PROBLEM, "No response from server", (Throwable)e);
        }
        finally {
            this.responseParams.addRequestTrace(request, response);
        }
    }

    private HttpResponseBuilder formatAccessTokenData() {
        HttpResponseBuilder builder = new HttpResponseBuilder();
        builder.addHeader("Content-Type", "application/json; charset=utf-8");
        builder.setHttpStatusCode(200);
        builder.setStrictNoCache();
        JSONObject json = new JSONObject(this.accessTokenData);
        builder.setResponseString(json.toString());
        return builder;
    }

    private void checkForProtocolProblem(HttpResponse response) throws OAuthProtocolException {
        if (this.isFullOAuthError(response)) {
            OAuthMessage message = this.parseAuthHeader(null, response);
            if (OAuthUtil.getParameter((OAuthMessage)message, (String)"oauth_problem") != null) {
                throw new OAuthProtocolException(message);
            }
            throw new OAuthProtocolException(response.getHttpStatusCode());
        }
    }

    private boolean isFullOAuthError(HttpResponse response) {
        if (response.getHttpStatusCode() != 401 && response.getHttpStatusCode() != 403) {
            return false;
        }
        if (this.realRequest.getOAuthArguments().mustUseToken()) {
            return true;
        }
        return this.accessorInfo.getAccessor().accessToken != null;
    }

    static List<Map.Entry<String, String>> selectOAuthParams(OAuthMessage message) {
        ArrayList result = Lists.newArrayList();
        for (Map.Entry param : OAuthUtil.getParameters((OAuthMessage)message)) {
            if (!OAuthFetcher.isContainerInjectedParameter((String)param.getKey())) continue;
            result.add(param);
        }
        return result;
    }

    private static boolean isContainerInjectedParameter(String key) {
        return (key = key.toLowerCase()).startsWith("oauth") || key.startsWith("xoauth") || key.startsWith("opensocial");
    }
}

