define("cq/widget/topics-list-view",
    [
        "jquery",
        "underscore",
        "backbone",
        "cq/util/events"
    ],
    function (
        $,
        _,
        Backbone,
        events) {

        return Backbone.View.extend({
            events: {
                "click .cq-edit": "editTopics"
            },

            initialize: function() {
                _.bindAll(this, "editTopics");

                // toggle icons
                var $icon = this.$(".aui-icon");
                this.model.on("request:topics", function () {
                    $icon.removeClass("aui-iconfont-edit").addClass("aui-icon-wait");
                });

                this.model.on("sync:topics", function () {
                    $icon.removeClass("aui-icon-wait").addClass("aui-iconfont-edit");
                });

                this.model.on("sync:topics", function () {
                    this.$el.hide();
                }, this);

                this.model.on("change:topics", this.updateTopics, this);
            },

            render: function() {
                return this;
            },

            updateTopics: function(model, topics) {
                // remove all items but the edit button
                this.$("li:not(.cq-edit)").remove();

                // format csv topics into objects
                topics = _.map(topics.split(","), function (topic) {
                    return { topic: {name: topic}};
                });

                var newTopicsHtml = _.map(topics, function (topic) {
                    return "<li>" + CQ.Templates.widget.topic(topic) + "</li>";
                }).join("");

                this.$el.prepend(newTopicsHtml);
            },

            editTopics: function() {
                this.trigger("edit");
            },

            show: function() {
                this.$el.show();
            }
        });
    });


define("cq/widget/topics-edit-view",
    [
        "jquery",
        "underscore",
        "backbone",
        "cq/util/events",
        "cq/widget/notifications",
        "cq/widget/topics-selector"
    ],
    function (
        $,
        _,
        Backbone,
        events,
        notifications,
        topicSelector) {

        return Backbone.View.extend({

            initialize: function() {
                _.bindAll(this, "cancelEdit");

                this.model.on("cq-request", function (model, xhr) {
                    this.hide();
                }, this);

                this.model.on("error", function (model, jqXhr) {
                    if (jqXhr.status === 401) {
                        var response = JSON.parse(jqXhr.responseText);
                        notifications.notify({
                            type: response.type || "warning", // use "warning" instead of "error" to adhere to ADG (a 4XX error should not lead to the destruction of data)
                            body: response.errorMessage,
                            errorKey: response.errorKey
                        });

                        var previousTopics = this.model.previousAttributes().topics;

                        this.model.set(this.model.previousAttributes()); // revert model back to it's previous state prior to modifications
                        if (previousTopics) {
                            this.$el.val(previousTopics); // revert input value
                        }
                    }
                }, this);
            },

            saveEdit: function() {
                var self = this;
                this.model.save({
                    topics: this.$el.val()
                }).error(function(jqXhr) {
                    self.trigger("error", self, jqXhr);
                });
            },

            hide: function() {
                this.$el.select2("destroy");
                this.trigger("hide");
            },

            cancelEdit: function() {
                // restore original values
                this.$el.val(this.model.get("topics"));
                this.hide();
            },

            /**
             * Setup select2 and make it behave nicely.
             *
             */
            render: function(suggestedTopics) {
                var self = this;

                topicSelector.initSelector(this.$el);

                var $s2container = this.$el.prev(".select2-container");
                var $input = $s2container.find(".select2-input");

                // set key listener for ESC, need to hack around select2
                // to not close when the dropdown is open
                var timeClosed = new Date();
                this.$el.on("select2-close", function() {
                    timeClosed = new Date();
                });
                $input.on("keydown", function (e) {
                    if (e.which === 27) { // escape
                        var now = new Date();
                        if (now - timeClosed > 200) {
                            self.cancelEdit();
                        }
                    }
                });

                // set key listener for RETURN, need to hack around select2
                // to not close when the dropdown is open
                var timeSelected = new Date();
                this.$el.on("select2-selecting", function () {
                    timeSelected = new Date();
                });
                $input.on("keydown", function (e) {
                    if (e.which === 13) { // enter
                        var s2dropdownOpen = $("#select2-drop").is(":visible");
                        var now = new Date();
                        if (!s2dropdownOpen && now - timeSelected > 200) {
                            self.saveEdit();
                        }
                    }
                });

                // wire up edit buttons
                $s2container.append(CQ.Templates.widget.topicEditButtons());
                $s2container.on("click", ".cq-save-button", function () {
                    self.saveEdit();
                });

                $s2container.on("click", ".cq-cancel-button", function () {
                    self.cancelEdit();
                });

                $input.focus();

                return this;
            }
        });
    });