define("cq/widget/questionslistmacro",
    [
        "ajs",
        "jquery",
        "underscore",
        "cq/widget/spacepicker",
        "exports"
    ], function (
        AJS,
        $,
        _,
        SpacePicker,
        exports
    ) {

    // Temporarily applied a new method of AJS.format then restores the old one
    // This is necessary because of a bug in 5.3 AUI that doesn't allow for choice statements
    function withFixedAJSFormat(fn) {
        var tmp = AJS.format;
        AJS.format = fixedAJSFormat;
        var result = fn();
        AJS.format = tmp;
        return result;
    }

    var fixedAJSFormat = function (message) {
        var apos = /'(?!')/g,
            simpleFormat = /^\d+$/,
            numberFormat = /^(\d+),number$/,
            choiceFormat = /^(\d+)\,choice\,(.+)/,
            choicePart = /^(\d+)([#<])(.+)/;

        var getParamValue = function (format, args) {
            var res = '', match;
            if (match = format.match(simpleFormat)) {
                res = args.length > ++format ? args[format] : '';
            }
            else if (match = format.match(numberFormat)) {
                res = args.length > ++match[1] ? args[match[1]] : '';
            }
            else if (match = format.match(choiceFormat)) {
                var value = (args.length > ++match[1] ? args[match[1]] : null);
                if (value !== null) {
                    var options = match[2].split('|');
                    var prevOptionValue = null;
                    for (var i = 0; i < options.length; i++) {
                        var parts = options[i].match(choicePart);
                        var argValue = parseInt(parts[1], 10);
                        if (value < argValue) {
                            if (prevOptionValue) {
                                res = prevOptionValue;
                                break;
                            } else {
                                res = parts[3];
                                break;
                            }
                        }
                        if (value == argValue && parts[2] == '#') {
                            res = parts[3];
                            break;
                        }
                        if (i == options.length - 1) {
                            res = parts[3];
                        }
                        prevOptionValue = parts[3];
                    }
                    var formatArgs = [res].concat(Array.prototype.slice.call(args, 1));
                    res = AJS.format.apply(AJS, formatArgs);
                }
            }
            return res;
        };

        var _performTokenRegex = function (message) {
            var tick = false, openIndex = -1, openCount = 0;
            for (var i = 0; i < message.length; i++) {
                var c = message.charAt(i);
                if (c == "'") {
                    tick = !tick;
                }
                if (tick) {
                    continue;
                }
                if (c === '{') {
                    if (openCount === 0) {
                        openIndex = i;
                    }
                    openCount++;
                }
                else if (c === '}') {
                    if (openCount > 0) {
                        openCount--;
                        if (openCount === 0) {
                            var match = [];
                            match.push(message.substring(0, i + 1));
                            match.push(message.substring(0, openIndex));
                            match.push(message.substring(openIndex + 1, i));
                            return match;
                        }
                    }
                }
            }
            return null;
        };

        var _format = function (message) {
            var args = arguments,
                res = "",
                match = _performTokenRegex(message);
            while (match) {
                message = message.substring(match[0].length);
                res += match[1].replace(apos, "");
                res += getParamValue(match[2], args);
                match = _performTokenRegex(message);
            }
            res += message.replace(apos, "");
            return res;
        };
        return _format.apply(AJS, arguments);
    };
    exports.onReady = function () {
        // Find all macros and their buttons (if they exist)
        var $macros = $(".cq-questions-list-macro");

        // Generate a link back to this page that we can use in question descriptions
        var pageId = AJS.Meta.get("page-id");
        if (pageId) {
            var pageLink = $('<a/>', {
                href: AJS.contextPath() + "/pages/viewpage.action?pageId=" + pageId,
                target: '_blank',
                text: AJS.Meta.get("page-title")
            }).prop("outerHTML");
        }

        // We apply the dialog to each macro so different question data doesn't persist to different macro dialogs
        _.each($macros, function (m) {
            var $m = $(m);
            // Bind analytics for button push (open ask question form)
            var $button = $m.find(".cq-ask-button");
            if ($button.length > 0) {
                $button.click(function () {
                    AJS.trigger("analytics", {name: "cq.questionslist-macro.ask-question-open"});
                });

                // Initialise the ask form window only when the user clicks the button
                $button.click(function() {
                    // Determine whether the macro we are coming from has a topic related to it
                    var contextTopics = $m.find(".cq-topics .cq-topic-label").map(function() {
                        return $(this).data("topic");
                    }).get();

                    // Bind the ask question dialog
                    var dialog = new AJS.Dialog({
                        width: 800,
                        height: 500,
                        id: "cq-ask-question-dialog",
                        closeOnOutsideClick: false
                    });

                    // Construct a form to display in the dialog
                    dialog.addHeader(AJS.I18n.getText('cq.question.ask'));

                    var templateParams = {};
                    if (contextTopics.length > 0) {
                        templateParams["topics"] = contextTopics.join(", ");
                    }

                    templateParams["spaceKey"] = AJS.Meta.get("space-key");

                    // Construct the ask question form via soy template
                    dialog.addPanel("Title",
                            $(CQ.Templates.widget.questionslistmacro.newQuestionForm(templateParams)),
                            "panel-body"
                    );
                    var $d = $("#" + dialog.id);

                    var $spaceKey = $d.find("#spaceKey");
                    $spaceKey.val(AJS.Meta.get("space-key"));
                    var spacePicker = new SpacePicker($spaceKey, AJS.I18n.getText("cq.question.ask.spacekey.placeholder"));

                    spacePicker.on("change", function () {
                        AJS.trigger("analytics", {name: "cq.questionslist-macro.space-picker-changed"});
                    });

                    // Bind the form submission to the submit button
                    dialog.addButton(AJS.I18n.getText('cq.question.ask.submit'), function (dialog) {
                        // Isolate the different components of the form
                        var $title = $d.find(".cq-macro-ask-title");
                        var $description = $d.find(".cq-macro-ask-description");
                        var $topics = $d.find(".cq-macro-ask-topics");
                        var titleVal = $title.find('input').val();
                        var descriptionVal = $description.find('textarea').val();

                        // Get the topics the question is being asked under
                        var topicsVal = $topics.find('input').val().split(/[ ,]+/);
                        // Remove duplicate topic entries
                        topicsVal = $.unique(topicsVal);
                        // Remove empty strings
                        topicsVal = _.reject(topicsVal, function (t) {  return !t });
                        // Convert our topic strings to "topic objects" for the question builder
                        var topicObjects = _.map(topicsVal, function (t) {
                            return {name: t};
                        });

                        // Assume there are no errors to begin with
                        var error = false;

                        // If we don't have a title, give an error
                        var $titleErrorField = $title.find(".error");
                        if (!titleVal) {
                            $titleErrorField.removeClass("hidden");
                            error = true;
                        } else {
                            $titleErrorField.addClass("hidden");
                        }

                        // If we don't have any topics, give an error
                        var $topicsErrorField = $topics.find(".error");
                        if (_.isEmpty(topicsVal)) {
                            $topicsErrorField.removeClass("hidden");
                            error = true;
                        } else {
                            $topicsErrorField.addClass("hidden");
                        }

                        if (!error) {
                            // Trigger analytics for actually asking a question from the popup form
                            AJS.trigger("analytics", {name: "cq.questionslist-macro.ask-question-click"});

                            // Inject a link back to this page in the question body
                            if (pageLink) {
                                descriptionVal += CQ.Templates.widget.questionslistmacro.contextLink({link: pageLink});
                            }

                            // Ask a question via the REST API
                            var newQuestion = {
                                'title': titleVal,
                                'body': descriptionVal,
                                'topics': topicObjects,
                                'spaceKey': $spaceKey.val()
                            };

                            // Ask the question using the REST API
                            var dfd = $.ajax({
                                type: "POST",
                                url: AJS.contextPath() + "/rest/questions/1.0/questions/",
                                data: JSON.stringify(newQuestion),
                                dataType: "json",
                                contentType: "application/json"
                            });

                            // Show the spinner and wait for completion
                            var $spinnerContainer = $(".spinner-container");
                            $spinnerContainer.spin({ lines: 12, length: 5, width: 3, radius: 8, trail: 60, speed: 1.5 });

                            dfd.done(function (data) {
                                // Append the details of the question to the existing questions list
                                var questionOptions = {
                                    idAsString: data.id.toString(),
                                    url: data.url,
                                    title: data.title,
                                    showMetadata: true,
                                    answers: data.answers,
                                    author: data.author,
                                    friendlyDateAsked: data.friendlyDateAsked,
                                    topics: data.topics,
                                    acceptedAnswerId: 0,
                                    votes: data.votes,
                                };
                                if (!!data.space) {
                                    questionOptions.space = {
                                        spaceName: data.space.spaceName,
                                        spaceKey: data.space.spaceKey
                                    }
                                }

                                // Use the overridden AJS.format to render the soy template for a question
                                var questionHTML = withFixedAJSFormat(function () {
                                    return CQ.Templates.feature.questionlist.question({question: questionOptions}, null, {});
                                });

                                // Add the newly-asked question to the macro list it came from
                                var $relatedList = $m.find("ol.cq-questionlist");
                                // If there are no questions currently in the list, empty it first (i.e. remove the empty message)
                                if (_.isEmpty($relatedList.children("li.cq-question"))) {
                                    $relatedList.empty();
                                }

                                // Highlight the question we just added
                                $(questionHTML).addClass("cq-highlighted").appendTo($relatedList);

                                // Close the dialog
                                dialog.remove();
                            });

                            // If asking a question fails, log an error in the console
                            dfd.fail(function () {
                                // Display an error in the dialog
                                AJS.messages.error("#cq-ask-form-error-banner", {
                                    title: AJS.I18n.getText('cq.macro.questionslist.ask.error.title'),
                                    body: AJS.I18n.getText('cq.macro.questionslist.ask.error.body')
                                });

                                console.error(AJS.I18n.getText('cq.macro.questionslist.ask.error.body'));
                            });

                            // Regardless of what happens after the request, stop the spinner when it is completed
                            dfd.always(function () {
                                $spinnerContainer.spinStop();
                            });

                        }

                    }, "aui-button");

                    dialog.addCancel(AJS.I18n.getText('cq.question.ask.cancel'), function (dialog) {
                        dialog.remove();
                    }, "#");

                    dialog.show();
                });
            }
        });
    };
});

require(["cq/widget/questionslistmacro"], function (QuestionsListMacro) {
    $(function () {
        QuestionsListMacro.onReady();
    });
});

