define("cq/widget/infinite-list",
    [
        "jquery",
        "underscore",
        "backbone",
        "cq/util/analytics",
        "cq/widget/loading",
        "exports"
    ],
    function (
        $,
        _,
        Backbone,
        analytics,
        loadingUtil,
        exports) {

        /**
         * Infinite scroll thingie. Loads more questions when you scroll down the page. Does that for a couple of times
         * and then displays a 'show more' button to load more content.
         *
         * Triggers 'after-render' event with a jquery collection of newly rendered items after a successful scroll load.
         */
        function ScrollLoader(options) {
            var self = this;
            var $loadMoreButton = $(CQ.Templates.widget.showMoreButton()).on("click", loadMoreItems);
            var loading = false;
            var loadCount = 0;
            var MAX_AUTO_LOAD = 2;

            var $list = options.$el;
            var loadDataFunction = options.load;
            var renderItem = options.render;
            var $container = options.$container || $(window);
            var body = options.body || document.body;
            var scrollMargin = options.scrollMargin || 300;
            var infinite = options.infinite || false;

            var ajaxPageSize = 15;
            var originalPageSize = $list.data("page-size");
            var startIndex = originalPageSize; // we start loading items at the page size offset

            function renderItems(items) {
                var itemHtml = items.map(function (item) {
                    return renderItem(item);
                }).join("");

                var $renderedItems = $(itemHtml);
                $list.append($renderedItems);

                self.trigger("after-render", $renderedItems);
            }

            function loadMoreItems() {
                loadCount++;
                $loadMoreButton.detach();

                var dfd = loadDataFunction(ajaxPageSize, startIndex);

                dfd.done(renderItems);

                dfd.done(function (items) {
                    startIndex += items.length;

                    var noMoreItemsToLoad = items.length < ajaxPageSize;
                    var exceededAutoLoad = !infinite && (loadCount >= MAX_AUTO_LOAD);

                    // remove scroll handler when we shouldn't auto load more items or there are no more items to be loaded
                    if (noMoreItemsToLoad || exceededAutoLoad) {
                        self.destroy();
                    }

                    // show load more button once we shouldn't auto load more items but there are still items to be loaded
                    if (exceededAutoLoad && !noMoreItemsToLoad) {
                        $list.append($loadMoreButton);
                    }
                });

                $list.attr("data-ajax-request", true);
                dfd.always(function () {
                    $list.attr("data-ajax-request", false);
                });

                loading = true;
                dfd.always(function () {
                    loading = false;
                });

                // loading spinner
                var $spinner = $("<div></div>").addClass("cq-spinner-container").css({"height": 50});
                $list.append($spinner);
                $spinner.spin(loadingUtil.SpinPresets.medium);
                dfd.always(function () {
                    $spinner.remove();
                });

                return dfd;
            }

            /**
             * Load items when we are almost scrolled to the bottom of a page and there are possibly more items
             * on the server to load (when we initially already display pageSize items).
             */
            function loadItemsMaybe() {
                if (loading) {
                    return;
                }

                var possiblyMoreItemsToLoad = $list.children().length >= originalPageSize;
                var scrolledToTheBottom = body.scrollHeight - $container.scrollTop() - scrollMargin  <= $container.height();
                if (possiblyMoreItemsToLoad && scrolledToTheBottom) {
                    loadMoreItems();
                }
            }

            $container.bind("scroll.list", _.debounce(loadItemsMaybe, 100));
            loadItemsMaybe();

            this.destroy = function() {
                $container.off("scroll.list");
            };
        }

        // mixin events
        _.extend(ScrollLoader.prototype, Backbone.Events);

        exports.init = function(options) {
            return new ScrollLoader(options);
        };
    });