define('cq-editor-loader/editor-loader', [
    'jquery',
    "cq/util/util"
], function(
    $,
    util
) {
    "use strict";

    var DEFAULT_LOADING_TIMEOUT = 12000; // ms

    var editorLoadingStatus;
    /**
     * An object that tracks events that should change the state of the editor which occur
     * when the editor is deactivated. These changes will apply to the editor when it is
     * next activated.
     */
    var stateChangeEventListener = {
        _listening: false,

        _queuedHandlers: [],

        _watchHandler: function() {
        },

        _unwatchHandler: function() {
        },

        _createQueueAdder: function(handler) {
            return function() {
                if (stateChangeEventListener._listening) {
                    stateChangeEventListener._queuedHandlers.push(handler);
                }
            };
        },

        /**
         * bind to all the relevant events.
         */
        bind: function() {
            AJS.bind("watchpage.pageoperation", this._createQueueAdder(this._watchHandler));
            AJS.bind("unwatchpage.pageoperation", this._createQueueAdder(this._unwatchHandler));
        },

        /**
         * @param listening if true then listen and queue handlers (i.e. the editor is deactivated). If false
         * then ignore any events. The editor is active and will handle them itself.
         */
        setListening: function(listening) {
            this._listening = listening;
        }
    };

    stateChangeEventListener.setListening(true);
    stateChangeEventListener.bind();

    /**
     * Check if a _load should be allowed and make the appropriate callback if it shouldn't.
     *
     * @return true if load is guarded (shouldn't be allowed); false if a load should be permitted.
     */
    var loadGuard = function(successCallback, errorCallback) {
        if (editorLoadingStatus) {
            editorLoadingStatus.fail(function() {
                if (errorCallback) {
                    errorCallback.call(this, arguments); // the .call() is optional, but if you want to pass promise data you'll need it
                } else {
                    AJS.log("EditorLoader: loadGuard - previous load failed.");
                }
            });

            editorLoadingStatus.done(function() {
                if (successCallback) {
                    //So that the promise has time to have callbacks registered.
                    editorLoadingStatus.done(function () {
                        setTimeout(successCallback, 0);
                    });
                } else {
                    AJS.log("EditorLoader: loadGuard - editor is already loaded.");
                }
            });

            return true;
        }
    };

    /**
     * Load the HTML and resources required by the Editor. (Note that the Editor is not necessarily initialised unless
     * the provided callback does it.)
     *
     * @param successCallback a function called if the Editor is successfully loaded.
     * @param errorCallback a function called if there is a failure while loading the Editor (takes a message string parameter).
     */
    var loadEditor = function(successCallback, errorCallback) {

        if (loadGuard(successCallback, errorCallback)) {
            return;
        }

        editorLoadingStatus = new $.Deferred();

        if (successCallback) {
            editorLoadingStatus.done(successCallback);
        }

        if (errorCallback) {
            editorLoadingStatus.fail(errorCallback);
        }

        // Create a hidden container to load the Editor DOM into
        var $preloadContainer = getPreloadContainer();
        $("body").append($preloadContainer);

        // Load the Editor template if on a page

        var editorTemplateDeferred = new $.Deferred();
        var onLoadCallback = function (response, status, xhr) {
            if (status === "success" || status === "notmodified") {
                // Put markup into dom
                var editorWrapper = $(response).find(".cq-editor");
                $preloadContainer.append(editorWrapper);
                AJS.loadTemplateScripts(editorWrapper);
                // move any metadata into the head (which is the only legal place for meta tags).
                var metadataTemplate = AJS.renderTemplate("dynamic-editor-metadata");
                $("head").append(metadataTemplate);
                AJS.populateParameters();
                AJS.debug("EditorLoader: Finished loading the editor template.");
                editorTemplateDeferred.resolve();
            } else {
                editorTemplateDeferred.reject("Error loading the Editor template: " + xhr.status + " - " + xhr.statusText);
            }
        };
        this.getEditorPreloadMarkup().done(onLoadCallback);

        var resourcesPromise = WRM.require(["wrc!editor", "wrc!macro-browser", "wrc!fullpage-editor"]).fail(function(err) {
            AJS.logError("Failed to load editor resources", err);
        });

        $.when(editorTemplateDeferred, resourcesPromise).done(function() {
            AJS.debug("EditorLoader: Finished loading the editor.");
            editorLoadingStatus.resolve();
        }).fail(function() {
            editorLoadingStatus.reject(arguments);
        });
    };

    /**
     * @returns the jQuery wrapped Element that contains the editor DOM, or create and return a new hidden div
     * if there is none found
     */
    var getPreloadContainer = function() {
        var $container = $("#editor-preload-container");
        if (!$container.length) {
            $container = $('<div id="editor-preload-container" style="display: none;"></div>');
        }

        return $container;
    };

    var cachedGetPreloadContentDeferred;

    return {

        /**
         * @returns the jQuery wrapped Element that contains the editor DOM, or create and return a new hidden div
         * if there is none found
         */
        getPreloadContainer : getPreloadContainer,

        getEditorPreloadMarkup: function (){
            if (cachedGetPreloadContentDeferred) {
                return cachedGetPreloadContentDeferred;
            }
            var editorAutoFocus = util.isIE();
            var url = AJS.contextPath() + "/questions/editor?contentId=" +  AJS.Meta.get('question-id') + "&editorAutoFocus=" + editorAutoFocus;
            // cache deferred.
            cachedGetPreloadContentDeferred = $.get(url, {
                atl_after_login_redirect: window.location.pathname, // the URL that an anonymous user will be redirect to after logging in,
                timeout: AJS.CQ.EditorLoader.loadingTimeout
            });
            return cachedGetPreloadContentDeferred;
        },

        /**
         * Check if the resources need to load the editor have been already loaded in the page.
         */
        resourcesLoaded: function () {
            return editorLoadingStatus && editorLoadingStatus.isResolved();
        },

        /** The maximum wait in milliseconds for the Editor to load */
        loadingTimeout: DEFAULT_LOADING_TIMEOUT,

        /**
         * @returns true if there is already an active editor; otherwise false
         */
        isEditorActive: function() {
            var $container = $("#editor-preload-container");
            return $container.length && $container.is(":visible");
        },

        /**
         * Load the Editor into a hidden Element on the page if it hasn't already been loaded.
         * The Editor is not initialised, its HTML, CSS and Javascript is simply loaded ready for
         * later activation.
         */
        load: loadEditor,

        /**
         * @return the immediate parent of the currently active editor as a jQuery wrapped Element. If an editor is not
         * currently active then null will be returned.
         */
        getEditorForm: function() {
            if (this.isEditorActive()) {
                return $(tinymce.activeEditor.getContainer()).closest('form');
            } else {
                return null;
            }
        }
    };
});

require('cq/module-exporter').exportModuleAsGlobal('cq-editor-loader/editor-loader', 'AJS.CQ.EditorLoader');
