define("cq/widget/announce",
    [
        "jquery",
        "underscore",
        "backbone",
        "cq/util/analytics"
    ],
    function ($, _, Backbone, analytics) {

        var Model = Backbone.Model.extend({
            url: AJS.contextPath() + "/rest/questions/1.0/announce",

            setMessage: function (message) {
                this.set("message", message, {silent: true});
            },

            validate: function (attrs) {
                if (!attrs.message) {
                    return {message: AJS.I18n.getText("cq.announce.error.no-message")};
                } else if (attrs.message.length > 1000) {
                    return {message: AJS.I18n.getText("cq.announce.error.message-too-long")};
                }
            },

            skipAnnouncement: function() {
                var dfd = $.ajax({
                    url: AJS.contextPath() + "/rest/questions/1.0/announce/skip",
                    type: "POST"
                });

                return dfd;
            }
        });

        var View = Backbone.View.extend({
            events: {
                "submit form": "formSubmit",
                "click .cq-close": "close",
                "click .cq-dismiss": "dismiss"
            },

            initialize: function(options) {
                var self = this;
                this.model.on("invalid", function (model, error) {
                    if (error.message) {
                        var $messageField = self.$(".cq-message-field");
                        $messageField.next(".error").remove();
                        $messageField.after(aui.form.fieldError({
                            message: error.message
                        }));
                        $messageField.focus();
                    }
                });
            },

            render: function() {
                this.$el.html(CQ.Templates.widget.announce.emailBannerDialog());
                return this;
            },


            formSubmit: function(e) {
                this.$(".error, .cq-dismiss").remove();
                var message = this.$(".cq-message-field").val();
                this.model.setMessage(message);

                if (this.model.save()) {
                    this.$(".cq-submit").attr("disabled", "disabled");
                    this.$(".cq-submit-result").html(CQ.Templates.widget.announce.emailsSending());

                    this.model.on("sync", function () {
                        this.$(".cq-submit-result").html(CQ.Templates.widget.announce.emailsSent());
                    }, this);

                    this.model.on("error", function () {
                        this.$(".cq-submit").attr("disabled", null);
                        this.$(".cq-submit-result").html(CQ.Templates.widget.announce.emailsError());
                    }, this);

                    analytics.trackEvent("announce", "mail-sent", {messageLength: message.length});
                }

                e.preventDefault();
            },

            close: function() {
                this.trigger("close");
                analytics.trackEvent("announce", "close");
            },

            dismiss: function(e) {
                e.preventDefault();

                this.model.skipAnnouncement();
                this.trigger("close");
                analytics.trackEvent("announce", "dismiss");
            }
        });

        function Announcement() {
            this.setMessage = function (msg) {
                this.model.setMessage(msg);
            };

            this.send = function () {
                this.model.save({}, {validate: true});
            };

            this.errors = function() {
                return this.model.validationError || [];
            };

            this.model = new Model();
            this.view = new View({model: this.model});
        }

        return Announcement;
    });


require(["jquery", "cq/widget/announce", "cq/util/analytics"], function ($, Announcement, analytics) {
    $(function () {
        $.ajax({
            url: AJS.contextPath() + "/rest/questions/1.0/announce/shouldShowBanner",
            type: "GET",
            cache: false,
            dataType: "json"
        }).done(function (result) {
            if (result == true) {
                $(".aui-page-panel-content").prepend(CQ.Templates.widget.announce.emailBanner());

                var $bannerOuter = $('.cq-announce-banner-outer');
                $('.cq-start', $bannerOuter).click(function () {
                    var announcement = new Announcement();

                    // close button
                    announcement.view.on("close", function () {
                        $bannerOuter.animate({height: 0}, function () {
                            $bannerOuter.remove();
                        });
                    });

                    // animate height via max-height trick
                    var height = $bannerOuter.height();
                    $bannerOuter.css("max-height", height);
                    $bannerOuter.find(".cq-announce-banner").html(announcement.view.render().$el);
                    $bannerOuter.css("max-height", "400px");

                    analytics.trackEvent("announce", "start");
                });
            }
        }); // we don't need an error callback handler here
    });
});