define("cq/feature/topiclist",
    [
        "jquery",
        "cq/api/topic-service",
        "cq/widget/infinite-list",
        "cq/widget/watch",
        "cq/widget/loading",
        "cq/util/analytics",
        "exports"
    ],
    function (
        $,
        topicService,
        infiniteList,
        Watch,
        loadingUtil,
        analytics,
        exports) {

        function bindWatchAndInitAnalytics($parent) {
            var $watchButtons = $parent.find(".cq-watch-button")
            $watchButtons.each(function() {
                var watch = new Watch({el: this});
                watch.render();
            });
            $watchButtons.on("click", function () {
                analytics.trackEvent("topiclist", "watchbuttonclick");
            });
        }

        function getTopics(pageSize, startIndex) {
            var query = {
                startIndex: startIndex,
                pageSize: pageSize,
                loadAvatar: "true"
            };
            return topicService.getTopics(query);
        }

        function getFeaturedTopics(pageSize, startIndex) {
            var query = {
                startIndex: startIndex,
                pageSize: pageSize,
                isFeatured: "true",
                loadAvatar: "true"
            };
            return topicService.getTopics(query);
        }

        function renderTopic(topic) {
            return CQ.Templates.feature.topiclist.topic({
                topicInfo: topic.toJSON(),
                canEditTopic: $("#cq-topiclist-outer").data("can-edit-topic"),
                showMetadata: true
            });
        }

        function makeDeleteDialog() {
            var dialog = AJS.InlineDialog($(".cq-delete-topic"), "delete-content", function (content, trigger, showPopup) {
                content.html(CQ.Templates.feature.topiclist.deleteTopic({
                    title: AJS.I18n.getText("cq.topic.delete.title"),
                    body: AJS.I18n.getText("cq.topic.delete.body")
                }));

                content.find(".cq-delete-cancel").on("click", function () {
                    dialog.hide();
                });

                content.find(".cq-delete-submit").on("click", function () {
                    var topicId = $(trigger).data("topic-id");
                    var dfd = topicService.deleteTopic(topicId);

                    // set disabled state
                    var $this = $(this);
                    $this.attr("disabled", "disabled").attr("data-ajax-request", true);
                    dfd.always(function () {
                        $this.attr("disabled", "").attr("data-ajax-request", false);
                    });

                    dfd.done(function () {
                        analytics.trackEvent("topic", "delete");
                        dialog.hide();

                        /**
                         * CQ-910: Remove topic straight from DOM (and avoid a page reload).
                         * If, after this removal, the user proceeds to the load the next page of results via infinitescroll,
                         * the next page of results will be missing 1 result. We decided that it's better to trade-off
                         * a single result for easier topic deletion.
                         */
                        $(trigger).closest(".cq-topic").remove();

                    });
                });

                showPopup();
            }, {
                useLiveEvents: true
            });

            return dialog;
        }

        function initTopicSearch() {
            var dfd, queryCount = 0, lastQuery;

            var $topicSearchField = $("#topic-search-input");
            $topicSearchField.keypress(function (event){
                if (event.keyCode === 13) {
                    event.preventDefault();
                    return false;
                }
            });

            $topicSearchField.keyup(function () {
                var $searchInput = $(this);

                if (queryCount++ === 0) {
                    $searchInput.addClass("search-in-progress");
                }

                var query = $searchInput.val();

                if (query == lastQuery) {
                    return;
                }

                var $topicList = $(".cq-all-topics");

                if (dfd) {
                    dfd.abort();
                }
                var canEditTopic = $("#cq-topiclist-outer").data("can-edit-topic");
                dfd = $.getJSON(AJS.contextPath() + "/rest/questions/1.0/topic/suggest?query=" + encodeURIComponent(query) + "&maxSuggestions=15");
                dfd.done(function (data) {
                    $topicList.empty();
                    $.each(data, function (i, topicInfo) {
                        $topicList.append(CQ.Templates.feature.topiclist.topic({
                            topicInfo: topicInfo,
                            canEditTopic: canEditTopic,
                            showMetadata: true
                        }));
                    });
                    $topicList.find(".cq-watch-button").each(function () {
                        new Watch({el: this}).render();
                    });
                    lastQuery = query;
                }).always(function () {
                    dfd = null;

                    if (--queryCount === 0) {
                        $searchInput.removeClass("search-in-progress");
                    }
                });
            });
        }

        exports.onReady = function() {
            $(function () {
                var $topicListOuter = $("#cq-topiclist-outer");
                var pageSize = $topicListOuter.data("page-size");
                var canEditTopic = $topicListOuter.data("can-edit-topic");

                var $featuredTopics = $("#cq-featured-topics-container");
                var dfdFeaturedTopics = getFeaturedTopics(pageSize, 0);
                var featuredTopicsSectionSpinner = new loadingUtil.SectionSpinner($featuredTopics, "cq-topics-section-loading-indicator");
                featuredTopicsSectionSpinner.showUntilResolved(dfdFeaturedTopics);
                dfdFeaturedTopics.done(function(topics) {
                    $featuredTopics.append(CQ.Templates.feature.topiclist.featuredTopics({"featuredTopics" : topics.toJSON(), "canEditTopic": canEditTopic}));
                    bindWatchAndInitAnalytics($featuredTopics);
                }).fail(function() {
                    $featuredTopics.append(CQ.Templates.feature.topiclist.featuredTopicsErrorMessage());
                });

                var $allTopicsContainer = $("#cq-all-topics-container");
                var dfdAllTopics = getTopics(pageSize, 0);
                var allTopicsSectionSpinner = new loadingUtil.SectionSpinner($allTopicsContainer, "cq-topics-section-loading-indicator");
                allTopicsSectionSpinner.showUntilResolved(dfdAllTopics);
                dfdAllTopics.done(function(topics) {
                    $allTopicsContainer.append(CQ.Templates.feature.topiclist.allTopicInfos({"topicInfos" : topics.toJSON(), "canEditTopic": canEditTopic}));
                    bindWatchAndInitAnalytics($allTopicsContainer);
                    var $allTopics = $allTopicsContainer.find(".cq-all-topics");
                    $allTopics.data("page-size", pageSize);
                    var loader = infiniteList.init({
                        $el: $allTopics,
                        load: getTopics,
                        render: renderTopic
                    });

                    loader.on("after-render", function($renderedItems) {
                        $renderedItems.find(".cq-watch-button").each(function () {
                            new Watch({el: this}).render();
                        });
                    });
                    initTopicSearch();
                }).fail(function() {
                    $allTopicsContainer.append(CQ.Templates.feature.topiclist.allTopicsErrorMessage());
                });
                makeDeleteDialog();
            });
        };
    });
