define("cq/feature/searchbox",
    [
        "jquery",
        "underscore",
        "backbone",
        "cq/api/search",
        "cq/widget/query-input",
        "cq/util/util",
        "cq/util/analytics",
        "ajs",
        "exports"
    ],
    function (
        $,
        _,
        Backbone,
        Search,
        QueryInput,
        util,
        analytics,
        AJS,
        exports) {

        var SearchboxView = QueryInput.extend({
            events: {
                blur: "onBlur",
                keydown: "onKeydown",
                keyup: "onKeyup"
            },

            initialize: function(options) {
                QueryInput.prototype.initialize.call(this);
                this.searchResultsView = options.searchResultsView;
            },

            onBlur: function() {
                this.searchResultsView.hide();
            },

            onKeydown: function(e) {
                if (e.which === 38) { //up
                    this.searchResultsView.prev();
                    e.preventDefault();
                } else if (e.which === 40) { // down
                    this.searchResultsView.next();
                    e.preventDefault();
                } else if (e.which === 13) { // enter
                    var $selected = this.searchResultsView.getSelected();
                    if ($selected.length) {
                        analytics.trackEvent("bsb", "click-result");
                        var href = $selected.find(".cq-question-link").attr("href");
                        window.location = href;
                        e.preventDefault();
                    }
                }
            },

            onKeyup: function(e) {
                if (e.which === 27) {
                    this.searchResultsView.hide();
                }
            }
        });

        var SearchResultView = Backbone.View.extend({
            events: {
                "mousedown" : "onMousedown"
            },

            render: function() {
                this.setElement(CQ.Templates.feature.searchbox.searchResult({question: this.model}));
                return this;
            },

            /**
             * CQ-641: Stop propagating the event so the SearchboxView does not receive the 'blur' event and then hides
             * the SearchResultsView which makes clicking BSB results impossible if you hold down the mouse button too
             * long.
             *
             */
            onMousedown: function(e) {
                e.stopPropagation();
                return false; // i know, but we need to return false otherwise the event keeps propagating (chrome, osx)
            }
        });

        var SearchResultsView = Backbone.View.extend({
            className: "cq-bsb-results",
            render: function() {
                if (!this.rendered) {
                    var $trigger = this.options.trigger;
                    this.$el.width($trigger.outerWidth());
                    $trigger.after(this.$el.html(CQ.Templates.feature.searchbox.searchResultsView()));
                    this.$el.hide();
                    this.rendered = true;
                }

                return this;
            },

            show: function (data, context) {
                if (data.search.spaceKey) {
                    this._showSpaceResults(data, context);
                } else {
                    this._showAllResults(data, context);
                }
            },

            _showSpaceResults: function (data, context) {
                if (!data.search.spaceKey) {
                    throw new Error("this function is for showing space results only.");
                }

                var results = data.results;

                this.render();

                var $list = this.$("ol").empty();

                $list.append(CQ.Templates.feature.searchbox.showingResultsInSpace({
                    spaceName: context.spaceName
                }));

                if (results.length === 0) {
                    $list.append(CQ.Templates.feature.searchbox.noResultsFound());
                } else {
                    _.each(results, function (result) {
                        $list.append(new SearchResultView({model: result}).render().el);
                    }, this);
                }

                if (data.lastPage) {
                    $list.append(CQ.Templates.feature.searchbox.searchForMoreResultsInAllSpaces({
                        query: data.search.expandedQuery
                    }));
                } else {
                    $list.append(CQ.Templates.feature.searchbox.searchForMoreResults({
                        query: data.search.expandedQuery,
                        spaceKey: data.search.spaceKey
                    }));
                }

                this.$el.fadeIn(100);
            },

            _showAllResults: function (data, context) {
                var results = data.results;

                this.render();

                var $list = this.$("ol").empty();

                if (context.spaceKey && results.length > 0) { // if we are showing results in the context of a space
                    $list.append(CQ.Templates.feature.searchbox.showingGlobalResults({
                        showSpaceResultsNotFoundMessage: context.spaceResultsFound === 0
                    }));
                }

                if (results.length === 0) {
                    $list.append(CQ.Templates.feature.searchbox.noResultsFound());
                } else {
                    _.each(results, function (result) {
                        $list.append(new SearchResultView({model: result}).render().el);
                    }, this);
                }

                if (!data.lastPage) {
                    $list.append(CQ.Templates.feature.searchbox.searchForMoreResults({
                        query: data.search.expandedQuery
                    }));
                }

                this.$el.fadeIn(100);
            },

            hide: function() {
                this.$el.fadeOut(100);
            },

            prev: function() {
                this.select(-1);
            },

            next: function() {
                this.select(1);
            },

            select: function(next) {
                var results = this.$("ol li").not(".results-description");
                var $active = results.filter(".cq-active");
                var index = results.index($active);
                $active.removeClass("cq-active");
                index += next;

                if (index < 0) {
                    index = results.length - 1;
                }

                if (index >= results.length) {
                    index = 0;
                }

                results.eq(index).addClass("cq-active");
            },

            getSelected: function() {
                return this.$("ol li.cq-active:visible");
            }
        });

        function focusInput() {
            var $input = $("#cq-searchbox .cq-bsb-input");
            $input.focus().val($input.val());
        }

        function initAnalytics(searchboxView, searchResultsView) {
            // track exact query only on EAC, privacy concerns.
            function getQueryData(query) {
                if (location.host === "extranet.atlassian.com") {
                    return { query: query };
                } else {
                    return { length: query.length, questionMark: query.indexOf("?") !== -1 };
                }
            }

            searchboxView.on("change", util.wait(function (value) {
                analytics.trackEvent("bsb", "search", getQueryData(value));
            }, 1000, this));

            searchResultsView.$el.on("click", ".cq-bsb-result", function () {
                analytics.trackEvent("bsb", "click-result");
            });

            $("#cq-searchbox").closest("form").on("submit", function () {
                var query = $(this).find(".cq-bsb-input").val();
                analytics.trackEvent("bsb", "ask-question", getQueryData(query));
            });
        }

        function init() {
            var search = new Search();

            var searchResultsView = new SearchResultsView({
                trigger: $("#cq-searchbox .cq-bsb-input")
            });

            var searchboxView = new SearchboxView({
                el: $("#cq-searchbox .cq-bsb-input"),
                searchResultsView: searchResultsView
            });

            // trigger collection search when searchbox changes
            searchboxView.on("change", util.wait(function (value) {
                if (value.length === 0) {
                    searchResultsView.hide();
                } else if (value.trim().length > 2) {
                    var spaceKey = AJS.Meta.get("space-key");

                    var params = {
                        query: value,
                        spaceKey: spaceKey
                    };

                    var context = {
                        spaceKey: spaceKey,
                        spaceName: AJS.Meta.get("space-name")
                    };

                    search.query(params, context);
                }
            }, 150, this));

            search.on("respond", function (data, context) {
                if (data.search.spaceKey && data.results.length === 0) { // if conducting a space search and there are no results, try searching for results globally
                    context.spaceResultsFound = 0;
                    search.query({ query: data.search.query }, context);
                } else {
                    searchResultsView.show(data, context);
                }
            });

            // loading spinner
            var spinner = $("#cq-searchbox .cq-bsb-icon");
            search.on("activity", function (activity) {
                if (activity) {
                    spinner.addClass("no-bg").spin();
                } else {
                    spinner.removeClass("no-bg").spin(false);
                }
            }, this);

            // focusInput();
            initAnalytics(searchboxView, searchResultsView);
        }

        exports.onReady = function() {
            $(init);
        }
    });
