define("cq/feature/questionlist",
    [
        "jquery",
        "underscore",
        "cq/widget/filters",
        "cq/widget/infinite-list",
        "cq/api/question-service",
        "cq/util/analytics",
        "cq/util/injecteddata",
        "exports",
        "ajs"
    ],
    function (
        $,
        _,
        filters,
        infiniteList,
        questionService,
        analytics,
        injectedData,
        exports,
        AJS) {

        function trackFilterView(currentFilter) {
            analytics.trackEvent("questionlist", "view-" + currentFilter);
        }

        function initAnalytics() {
            $(".cq-watch-button").on("click", function () {
                analytics.trackEvent("questionlist", "watchbuttonclick");
            });

            _.defer(function() {
                trackFilterView(filters.getActiveFilter());
            });

            $(document).on("click", ".cq-topic-label", function () {
                analytics.trackEvent("questionlist", "topiclabelclick");
            });
        }

        /**
         * Build the ajax query data for questions
         * @param ajaxPageSize
         * @param startIndex
         * @returns {{startIndex: *, pageSize: *, filter: *}}
         */
        function buildQuery(ajaxPageSize, startIndex) {
            var $questionList = $(".cq-questionlist");

            var filter = $questionList.attr("data-filter");
            var topicId = $questionList.attr("data-topic-id");
            var spaceKey = $questionList.attr("data-space-key");

            var query = {
                startIndex: startIndex,
                pageSize: ajaxPageSize,
                filter: filter
            };

            if (topicId) {
                _.extend(query, { topicId: topicId });
            }

            if (spaceKey) {
                _.extend(query, { spaceKey: spaceKey });
            }

            return query;
        }

        /**
         * Privately used by the scrollLoader
         * @param ajaxPageSize
         * @param startIndex
         * @returns {$.Deferred}
         */
        function getQuestions(ajaxPageSize, startIndex) {
            return questionService.getQuestions(buildQuery(ajaxPageSize, startIndex));
        }

        /**
         * Privately used by the scrollLoader
         * @param question
         * @returns {*}
         */
        function renderData(question) {
            return CQ.Templates.feature.questionlist.question({question: question.toJSON()}, null, injectedData.get());
        }

        var scrollLoader = null;
        function initScrollLoader() {
            if (scrollLoader) {
                scrollLoader.destroy();
            }

            scrollLoader = infiniteList.init({
                $el: $(".cq-questionlist"),
                load: getQuestions,
                render: renderData
            });
        }

        /**
         * Query and render questions
         * @param query
         * @param $questionlist
         * @param href
         */
        function renderQuestions(query, $questionlist, href) {
            // get and render questions
            var dfd = questionService.getQuestions(query);

            dfd.done(function (questions) {
                var state = {
                    questions: questions.toJSON(),
                    filter: query.filter
                };

                render(state, $questionlist);
                var pushStateSupport = "pushState" in window.history;
                if (!!href && pushStateSupport) {
                    history.pushState(state, "", href);
                }
            });

            dfd.fail(function() {
                // Render an error message
                $questionlist.html(CQ.Templates.feature.questionlist.errorMessage());
            });

            // analytics
            dfd.always(function() {
                trackFilterView(query.filter);
            });
            var $spinner = $(".cq-filters .filter-control .aui-icon-wait");
            // spinner
            $spinner.show();
            dfd.always(function () {
                $spinner.hide();
            });

            $questionlist.attr("data-ajax-request", true);
            dfd.always(function () {
                $questionlist.attr("data-ajax-request", false);
            });
        }

        /**
         * Render the question based on the state
         * @param state
         * @param $questionlist
         */
        function render(state, $questionlist) {
            var html = CQ.Templates.feature.questionlist.questions({
                questions: state.questions,
                emptyMessage: AJS.I18n.getText('cq.questions.list.empty')
            }, null, injectedData.get());

            $questionlist.html(html).attr("data-filter", state.filter);

            initScrollLoader();
        }

        /**
         * Init filter switching when you click on popular/recent/etc filters
         */
        function initFilterSwitch() {
            var $questionlist = $(".cq-questionlist");
            var PAGE_SIZE = $questionlist.attr("data-page-size");
            var query = buildQuery(PAGE_SIZE, 0);

            // spinner
            var $spinner = $("<div class='aui-icon aui-icon-wait'></div>").css({"margin-top": "4px", "float": "left"}).hide();
            $(".cq-filters .filter-control").append($spinner);

            filters.on("change", function (filter, href, e) {
                e.preventDefault();
                // get and render questions
                query.filter = filter;
                renderQuestions(query, $questionlist, href);
            });

            query.filter = $questionlist.attr("data-filter");
            renderQuestions(query, $questionlist);
        }

        /**
         * Cheap and easy implementation of AJAXy filter switching when you click on popular/recent/etc filters. Keeps
         * back/forward buttons working with pushState.
         */
        function initHandlePushState() {
            var pushStateSupport = "pushState" in window.history;
            if (pushStateSupport) {
                // handle back/forward buttons history
                var popped = ('state' in window.history && window.history.state !== null), initialURL = location.href;
                $(window).on("popstate", function (e) {
                    // Ignore inital popstate that some browsers fire on page load
                    // http://stackoverflow.com/questions/6421769/popstate-on-pages-load-in-chrome
                    var initialPop = !popped && location.href == initialURL;
                    popped = true;
                    if (initialPop) {
                        return;
                    }

                    var state = e.originalEvent.state;
                    if (state == null) {
                        // we don't have state if we go back to the initial page load state, so just reload
                        window.location.reload();
                        return;
                    }

                    // render list and set filters
                    render(state, $questionlist);
                    filters.setFilter(state.filter);
                    $("body").scrollTop(0);
                });
            }
        }

        exports.onReady = function() {
            $(function () {
                initScrollLoader();
                initAnalytics();
                initFilterSwitch();
                initHandlePushState();
            });
        };
    });